%% Computing time derivatives of state variables
% this is a stripped down version of the TCA cycle model


function varargout = DAEs(t,x,p)
% initialize LHS
f = zeros(size(x,1),size(x,2));

%% Compute binding polynomial array
p.BP = ones(size(x,1),size(x,2));
p.BP(p.i.Lmtx,:) = BP(p.if.ligands,'mtx',x,p);
p.BP(p.i.Lcyt,:) = BP(p.if.ligands,'cyt',x,p);

%% Compute Algebriac Constraint Residuals

% Compute other Matrix Conserved Species
f(p.i.ATP_x,:) = x(p.i.ATP_x,:) - (p.Atot - x(p.i.ADP_x,:));
f(p.i.NADH_x,:) = x(p.i.NADH_x,:) - (p.Ntot - x(p.i.NAD_x,:));
f(p.i.UQH2_x,:) = x(p.i.UQH2_x,:) - (p.Qtot - x(p.i.UQ_x,:));
f(p.i.c2_x,:) = x(p.i.c2_x,:) - (p.Ctot - x(p.i.c3_x,:));
f(p.i.PCr_c,:) = x(p.i.PCr_c,:) - (p.Crtot - x(p.i.Cr_c,:));

% for Calcium_Loading
fCai = p.CaPi.Ki./(x(p.i.CaPi,:) + p.CaPi.Ki); % CaPi inhibition

% p.MitoDH.x = p.MitoDH.x.*fCai;
p.CI.x = p.CI.x.*fCai;

%% Compute TCA Reaction Rates (nmol/min/mg)
% this version assumes all PDH with only 25% SDH. Calcium activates TCA and
% brings SDH contribution to 1 QH2 for every 4 NADH

J.MitoDH_carb = MitoDHm13(x,p);   % NAD+ = NADH + H+ w/ michaelis-menten ADP*Pi/ATP and NAD/NADH feedback and with Hill coeff for NAD
J.MitoDH_fat = 0;
J.ME = p.kf_ME.*x(p.i.MAL_x,:).*x(p.i.NAD_x,:); % malic enzyme (decarboxylating)

%% Compute ETS Fluxes (nmol/min/mg)

[J.CI.JNADH,J.CI.JQ,J.CI.JSO,J.CI.JH2O2,J.CI.d] = ETC1m3(x,p);       % NADH + UQ + 5Hmtx+ = NAD + UQH2 + 4Hims+

% ----- Complex II -----
[J.CII.JSUC, J.CII.JSO, J.CII.JH2O2, J.CII.JQH2, ~, J.CII.Xinact, J.CII.d] = CII_v2(x,p);

[J.CIII.Jc3, J.CIII.JQp, J.CIII.JQn, J.CIII.JSO,J.CIII.d] = ETC3m2(x,p);   % UQH2 + 2Cytc(ox) = UQ + 2Cytc(red) + 2Hims+

J.CIV = ETC4m1(x,p);     % 2Cytc(red) + 1/2O2 + 4Hmtx+ = 2Cytc(ox) + H20 + 2Hims+

J.mHleak = HLEAKm4(x,p);

%% DCC
J.DCC = DCCm2(x,p);

%% FH
J.FH = FH(x,p);

%% MDH
J.MDH = MDH(x,p);

%% MCT

J.MCT = MCT(x,p);

%% Compute ATP/ADP cycle Fluxes

J.F1FO = F1FOATPASEm0(x,p);     % ADP + Pi + 3Hims+ = ATP + 2Hims+
J.ANT = ANTm2(x,p);   
J.PiC = PIHm1(x,p); 

%% Compute K+ Cycle Fluxes

J.mKleak = KLEAKm0(x,p);
J.mKHE = KHEm0(x,p);

%% Na+/Ca+ Cycle Fluxes

J.MCU = MCUm1(x,p);
J.NCLX = NCLXm6(x,p);
J.mNHE = NHEm1(x,p);
J.mCHE2 = 0;%CHE2m0(x,p); % electroneutral CHE with 2:1 stoichiometry
J.mCHE3 = 0;%CHE3m0(x,p); % electrogenic CHE with 3:1 stoichiometry
J.RaM = 0;%RaMm0(x,p);    % electrogenic RaM, not needed for these sims

%% Calcium Phosphate Buffering

J.CaPi = CaPim0(x,p,p.CaPi.kf,p.CaPi.K,p.CaPi.n,p.CaPi.m); % non-saturable
% J.CaPi = CaPim1(x,p,p.CaPi.kf,p.CaPi.K,p.CaPi.n,p.CaPi.m,p.CaPi.Btot); % saturable

%% Adenylate Kinase

% J.imAK = 0; %AKm0(x,p); % not present in BISEN example .bsl files

%% ROS Scavenging (currently, place-holders)
% rates derived from values in Murphy's 2012 ARS paper and his 2010 Biochem
% J paper
% b/c Etot for CI and CIII about X higher than reported, increasing
% scavenging fluxes by X
J.SODx = 2e4*60*x(p.i.SO_x,:)*1; % assuming 10uM MnSOD and k of 2e9 M^-1*s^1 (M/s)
J.SODc = 2e4*60*x(p.i.SO_c,:); % assuming similiar rate for matrix (M/s)
J.H2O2scavenge = p.kf_scavenging*x(p.i.H2O2_x,:)./(p.Km_H2O2 + x(p.i.H2O2_x,:)); % saturable
J.H2O2perm = 333*60*(x(p.i.H2O2_x,:) - x(p.i.H2O2_c,:))*p.xEmission; % based information in Bienert et al. 2006 BBA review and 6 nm membrane thickness (2 um/s / 6 nm) wrt to matrix volume
J.HrP = 1e6*x(p.i.H2O2_c,:); % approx

%% Compute Time Derivatives for Matrix non-cation State Variables
J.FCCP = p.FCCP.x1*(x(p.i.dPsi,:) - 0);
J.FCCP2 = p.FCCP.x2*(x(p.i.H_c,:)./x(p.i.H_x) - 1);
% dPsi
f(p.i.dPsi,:) = (+ 4*J.CI.JNADH + 1*J.CIII.Jc3 + 4*J.CIV - p.F1FO.nH*J.F1FO - 1*J.ANT - 1*J.mHleak - 1*J.mKleak - 2*J.MCU - 1*J.NCLX - 1*J.FCCP)/p.Cmito;
% Matrix species
f(p.i.ADP_x,:) = (+ 1*J.ANT - 1*J.F1FO - 1*J.MitoDH_carb - 1*J.MitoDH_fat)/p.Vol.H2O_mtx;
f(p.i.Pi_x,:) = (+ 1*J.PiC + 1*J.DCC.Pi - 1*J.F1FO - 1*J.MitoDH_carb - 1*J.MitoDH_fat)/p.Vol.H2O_mtx - p.CaPi.m*J.CaPi;
f(p.i.NAD_x,:) = (-3*J.MitoDH_carb - 3*J.MitoDH_fat - 1*J.MDH + 1*J.CI.JNADH  - 1*J.ME)/p.Vol.H2O_mtx;
f(p.i.UQ_x,:) = (- 1*J.CI.JNADH + 1/2*J.CIII.Jc3 - 1*J.CII.JQH2)/p.Vol.lipid_imm;
f(p.i.SO_x,:) = (+ 1*J.CI.JSO + 1/2*J.CIII.JSO + 1*J.CII.JSO)/p.Vol.H2O_mtx - 2*J.SODx;
f(p.i.H2O2_x,:) = (+ 1*J.CI.JH2O2 + 1*J.CII.JH2O2)/p.Vol.H2O_mtx + 1*J.SODx - 1*J.H2O2scavenge - 1*J.H2O2perm;
f(p.i.O2_x,:) = -J.CIV/2/p.Vol.H2O_cyt + 2e-8; % allow small amount of back diffusion to prevent negative O2 (add O2 consumption by ROS rxns)
f(p.i.CaPi,:) = J.CaPi; % calcium phosphate
f(p.i.PYR_x,:) = (J.MCT - J.MitoDH_carb  + J.ME)/p.Vol.H2O_mtx;
f(p.i.SUC_x,:) = (J.DCC.SUC - J.CII.JSUC + J.MitoDH_carb)/p.Vol.H2O_mtx;
f(p.i.FUM_x,:) = (J.CII.JSUC - J.FH)/p.Vol.H2O_mtx;
f(p.i.MAL_x,:) = (J.FH + J.DCC.MAL - J.MDH  - J.ME)/p.Vol.H2O_mtx;
f(p.i.OAA_x,:) = (J.MDH - J.MitoDH_carb)/p.Vol.H2O_mtx; % OAA consumed via CS enabled by ME and PDH rxns

%% Compute Matrix Cation Derivatives Based on Rapid Equilibrium Dynamics

f = mCation_dXdt(x,f,p,J);

%% Compute Time Derivatives for IMS non-cation State Variables

f(p.i.c3_x,:) = (- 1*J.CIII.Jc3 + 2*J.CIV)/p.Vol.H2O_ims;

%% Reactant Injection (adding bolus amounts to buffer)
J = InjectReactants(t,x,p,J);

%% External ATPase

J.ATPase = ATPasem1(x,p); % ATPase for Vinnakota et al. 2015
J.CK=zeros(1,size(x,2));
J.AK=zeros(1,size(x,2));

%% Compute Derivative for cytosolic/buffer space

% Buffer/Cytosolic species
f(p.i.Pi_c,:) = (+ 1*J.Inject(p.if.Pi,:) - 1*J.PiC - J.DCC.Pi)/p.Vol.H2O_cyt + 1*J.ATPase;
f(p.i.ATP_c,:) = (+ 1*J.Inject(p.if.ATP,:) + 1*J.ANT)/p.Vol.H2O_cyt - 1*J.ATPase + 1*J.AK + 1*J.CK;
f(p.i.ADP_c,:) = (+ 1*J.Inject(p.if.ADP,:) - 1*J.ANT)/p.Vol.H2O_cyt + 1*J.ATPase - 2*1*J.AK - 1*J.CK;
f(p.i.AMP_c,:) = 1*J.AK;
f(p.i.Cr_c,:) = 1*J.CK;
f(p.i.SO_c,:) = (+ 1/2*J.CIII.JSO)/p.Vol.H2O_cyt  - 2*J.SODc;
f(p.i.H2O2_c,:) = J.SODc + 1*J.H2O2perm*p.Vol.H2O_mtx/p.Vol.H2O_cyt - J.HrP; % H2O2perm in units of M/s wrt matrix H2O, H2O_cyt in units of l buffer/lmito, H2O_mtx converts l buffer/lmito to lbuffer/lH2Omtx
f(p.i.Reso,:) = J.HrP;
f(p.i.PYR_c,:) = (-J.MCT)/p.Vol.H2O_cyt;
f(p.i.SUC_c,:) = -(J.DCC.SUC)/p.Vol.H2O_cyt;
f(p.i.MAL_c,:) = -(J.DCC.MAL)/p.Vol.H2O_cyt;

%% Compute Cytosolic Cation Derivatives Based on Rapid Equilibrium Dynamics
f = cCation_dXdt(x,f,p,J);

%% Clamp buffer/cytosolic concentrations to generate steady state
switch p.exp
case 0 
f(p.i.cyt,:) = 0; % clamp all buffer variables
case 1
f([p.i.H_c,p.i.K_c,p.i.Na_c,p.i.Mg_c,p.i.Ca_c],:)=0; % clamp all buffer free cation variables
case 2
f(p.i.H_c,:) = 0; % only fix buffer pH
end
%% extra output arguments
varargout={f,J};
end

%% Compute Ion Derivatives Using Rapid Equilibrium Approximation Method

function f = mCation_dXdt(x,f,p,J)
%% Compute Time Derivatives for Cations
H_x = x(p.i.H_x,:);     % M
K_x = x(p.i.K_x,:);     % M
Na_x = x(p.i.Na_x,:);   % M
Mg_x = x(p.i.Mg_x,:);   % M
Ca_x = x(p.i.Ca_x,:);   % M

%% Define Ligand Vector
Li = x(p.i.Lmtx,:); % M

%% Define Ligand Derivatives
dLidt = f(p.i.Lmtx,:); % M/s
% set dLidt for algebraic constraints equal to negative of counterpart
dLidt(1,:) = -f(p.i.ADP_x,:); % indices hard coded
dLidt(5,:) = -f(p.i.NAD_x,:); % indices hard coded

%% Partial Derivatives of binding polynomials:
Pm_L = p.BP(p.i.Lmtx,:);
Pm_L2 = Pm_L.^2;
Pm_HBH = (1 + 1./p.KD(p.if.ligands,p.if.K)*K_x + 1./p.KD(p.if.ligands,p.if.Na)*Na_x ...
    + 1./p.KD(p.if.ligands,p.if.Mg)*Mg_x + 1./p.KD(p.if.ligands,p.if.Ca)*Ca_x);
Pm_KBK = (1 + 1./p.KD(p.if.ligands,p.if.H)*H_x + 1./p.KD(p.if.ligands,p.if.Na)*Na_x ...
    + 1./p.KD(p.if.ligands,p.if.Mg)*Mg_x + 1./p.KD(p.if.ligands,p.if.Ca)*Ca_x);
Pm_NBN = (1 + 1./p.KD(p.if.ligands,p.if.H)*H_x + 1./p.KD(p.if.ligands,p.if.K)*K_x ...
    + 1./p.KD(p.if.ligands,p.if.Mg)*Mg_x + 1./p.KD(p.if.ligands,p.if.Ca)*Ca_x);
Pm_MBM = (1 + 1./p.KD(p.if.ligands,p.if.H)*H_x + 1./p.KD(p.if.ligands,p.if.K)*K_x ...
    + 1./p.KD(p.if.ligands,p.if.Na)*Na_x + 1./p.KD(p.if.ligands,p.if.Ca)*Ca_x);
Pm_CBC = (1 + 1./p.KD(p.if.ligands,p.if.H)*H_x + 1./p.KD(p.if.ligands,p.if.K)*K_x ...
    + 1./p.KD(p.if.ligands,p.if.Na)*Na_x + 1./p.KD(p.if.ligands,p.if.Mg)*Mg_x);

pHBpH = + sum(Pm_HBH.*Li./Pm_L2./(p.KD(p.if.ligands,p.if.H)*ones(1,size(Li,2))),1);
pHBpK = - H_x.*sum(Li./Pm_L2 ...
    ./(p.KD(p.if.ligands,p.if.H).*p.KD(p.if.ligands,p.if.K)*ones(1,size(Li,2))),1);
pHBpNa = - H_x.*sum(Li./Pm_L2 ...
    ./(p.KD(p.if.ligands,p.if.H).*p.KD(p.if.ligands,p.if.Na)*ones(1,size(Li,2))),1);
pHBpMg = - H_x.*sum(Li./Pm_L2 ...
    ./(p.KD(p.if.ligands,p.if.H).*p.KD(p.if.ligands,p.if.Mg)*ones(1,size(Li,2))),1);
pHBpCa = - H_x.*sum(Li./Pm_L2 ...
    ./(p.KD(p.if.ligands,p.if.H).*p.KD(p.if.ligands,p.if.Ca)*ones(1,size(Li,2))),1);

pKBpH = - K_x.*sum(Li./Pm_L2 ...
    ./(p.KD(p.if.ligands,p.if.K).*p.KD(p.if.ligands,p.if.H)*ones(1,size(Li,2))),1);
pKBpK = + sum(Pm_KBK.*Li./Pm_L2./(p.KD(p.if.ligands,p.if.K)*ones(1,size(Li,2))),1);
pKBpNa = - K_x.*sum(Li./Pm_L2 ...
    ./(p.KD(p.if.ligands,p.if.K).*p.KD(p.if.ligands,p.if.Na)*ones(1,size(Li,2))),1);
pKBpMg = - K_x.*sum(Li./Pm_L2 ...
    ./(p.KD(p.if.ligands,p.if.K).*p.KD(p.if.ligands,p.if.Mg)*ones(1,size(Li,2))),1);
pKBpCa = - K_x.*sum(Li./Pm_L2 ...
    ./(p.KD(p.if.ligands,p.if.K).*p.KD(p.if.ligands,p.if.Ca)*ones(1,size(Li,2))),1);

pNaBpH = - Na_x.*sum(Li./Pm_L2 ...
    ./(p.KD(p.if.ligands,p.if.Na).*p.KD(p.if.ligands,p.if.H)*ones(1,size(Li,2))),1);
pNaBpK = - Na_x.*sum(Li./Pm_L2 ...
    ./(p.KD(p.if.ligands,p.if.Na).*p.KD(p.if.ligands,p.if.K)*ones(1,size(Li,2))),1);
pNaBpNa = + sum(Pm_NBN.*Li./Pm_L2./(p.KD(p.if.ligands,p.if.Na)*ones(1,size(Li,2))),1);
pNaBpMg = - Na_x.*sum(Li./Pm_L2 ...
    ./(p.KD(p.if.ligands,p.if.Na).*p.KD(p.if.ligands,p.if.Mg)*ones(1,size(Li,2))),1);
pNaBpCa = - Na_x.*sum(Li./Pm_L2 ...
    ./(p.KD(p.if.ligands,p.if.Na).*p.KD(p.if.ligands,p.if.Ca)*ones(1,size(Li,2))),1);

pMgBpH = - Mg_x.*sum(Li./Pm_L2 ...
    ./(p.KD(p.if.ligands,p.if.Mg).*p.KD(p.if.ligands,p.if.H)*ones(1,size(Li,2))),1);
pMgBpK = - Mg_x.*sum(Li./Pm_L2 ...
    ./(p.KD(p.if.ligands,p.if.Mg).*p.KD(p.if.ligands,p.if.K)*ones(1,size(Li,2))),1);
pMgBpNa = - Mg_x.*sum(Li./Pm_L2 ...
    ./(p.KD(p.if.ligands,p.if.Mg).*p.KD(p.if.ligands,p.if.Na)*ones(1,size(Li,2))),1);
pMgBpMg = + sum(Pm_MBM.*Li./Pm_L2./(p.KD(p.if.ligands,p.if.Mg)*ones(1,size(Li,2))),1);
pMgBpCa = - Mg_x.*sum(Li./Pm_L2 ...
    ./(p.KD(p.if.ligands,p.if.Mg).*p.KD(p.if.ligands,p.if.Ca)*ones(1,size(Li,2))),1);

pCaBpH = - Ca_x.*sum(Li./Pm_L2 ...
    ./(p.KD(p.if.ligands,p.if.Ca).*p.KD(p.if.ligands,p.if.H)*ones(1,size(Li,2))),1);
pCaBpK = - Ca_x.*sum(Li./Pm_L2 ...
    ./(p.KD(p.if.ligands,p.if.Ca).*p.KD(p.if.ligands,p.if.K)*ones(1,size(Li,2))),1);
pCaBpNa = - Ca_x.*sum(Li./Pm_L2 ...
    ./(p.KD(p.if.ligands,p.if.Ca).*p.KD(p.if.ligands,p.if.Na)*ones(1,size(Li,2))),1);
pCaBpMg = - Ca_x.*sum(Li./Pm_L2 ...
    ./(p.KD(p.if.ligands,p.if.Ca).*p.KD(p.if.ligands,p.if.Mg)*ones(1,size(Li,2))),1);
pCaBpCa = + sum(Pm_CBC.*Li./Pm_L2 ...
    ./(p.KD(p.if.ligands,p.if.Ca)*ones(1,size(Li,2))),1);

%% PHI's:
PhiH = - H_x.*sum(dLidt./Pm_L ...
    ./(p.KD(p.if.ligands,p.if.H)*ones(1,size(Li,2)))) ...
    + ((1).*(J.MitoDH_fat + J.MitoDH_carb) + 1*J.MCT + 1*J.MDH - 5*J.CI.JNADH - 1*J.CIII.Jc3 - 4*J.CIV ... % see Wu et al. 2007, net number of protons released per complete TCA cycle including PYR import = 3
    + (p.F1FO.nH-1)*J.F1FO + 2*J.PiC + J.mHleak - 0*J.H2O2scavenge...
    + J.mKHE + J.mNHE - 0*J.CI.JH2O2 - 0*J.SODx + J.FCCP + J.FCCP2 + 2*J.mCHE2 + 3*J.mCHE3 + 0*J.ME)/p.Vol.H2O_mtx + 2*(p.CaPi.n - p.CaPi.m)*J.CaPi; %
PhiK = - K_x.*sum(dLidt./Pm_L ...
    ./(p.KD(p.if.ligands,p.if.K)*ones(1,size(Li,2)))) ...
    + (J.mKleak - J.mKHE)/p.Vol.H2O_mtx;
PhiNa = - Na_x.*sum(dLidt./Pm_L ...
    ./(p.KD(p.if.ligands,p.if.Na)*ones(1,size(Li,2))))...
    + (3*J.NCLX - J.mNHE)/p.Vol.H2O_mtx;
PhiMg = - Mg_x.*sum(dLidt./Pm_L ...
    ./(p.KD(p.if.ligands,p.if.Mg)*ones(1,size(Li,2))));
PhiCa = - Ca_x.*sum(dLidt./Pm_L ...
    ./(p.KD(p.if.ligands,p.if.Ca)*ones(1,size(Li,2)))) ...
    + (J.MCU - J.NCLX - J.mCHE2 - J.mCHE3 + J.RaM)/p.Vol.H2O_mtx - p.CaPi.n*J.CaPi; % add CHE

%% Endogenous buffering for [H+], [K+], [Na+], [Mg2+], [Ca2+], [Pi]
BH = p.Buff.Htot;       % M
K_BH = p.Buff.KD_H;     % M

BK = p.Buff.Ktot;       % M
K_BK = p.Buff.KD_K;     % M

BNa = p.Buff.Natot;     % M
K_BNa = p.Buff.KD_Na;   % M

BMg = p.Buff.Mgtot;     % M
K_BMg = p.Buff.KD_Mg;   % M

BCa = p.Buff.BCa;      % M
K_BCa = p.Buff.K_BCa;  % M

%% Buffering Terms:
DH = 1 + pHBpH + BH./K_BH./(1+H_x./K_BH).^2;
DK = 1 + pKBpK + BK./K_BK./(1+K_x./K_BK).^2;
DNa = 1 + pNaBpNa + BNa./K_BNa./(1+Na_x./K_BNa).^2;
DMg = 1 + pMgBpMg + BMg./K_BMg./(1+Mg_x./K_BMg).^2;
DCa = 1 + pCaBpCa + BCa./K_BCa./(1+Ca_x./K_BCa).^2;

%% Solution to linear cation eqns.

D=(DH.*DK.*DMg.*pCaBpNa.*pNaBpCa + DH.*DK.*DNa.*pCaBpMg.*pMgBpCa + DCa.*DH.*DK.*pMgBpNa.*pNaBpMg + DK.*DMg.*DNa.*pCaBpH.*pHBpCa + DH.*DMg.*DNa.*pCaBpK.*pKBpCa + DCa.*DK.*DMg.*pHBpNa.*pNaBpH + DCa.*DH.*DMg.*pKBpNa.*pNaBpK + DCa.*DK.*DNa.*pHBpMg.*pMgBpH + DCa.*DH.*DNa.*pKBpMg.*pMgBpK + DCa.*DMg.*DNa.*pHBpK.*pKBpH - DH.*DK.*pCaBpNa.*pMgBpCa.*pNaBpMg - DH.*DK.*pCaBpMg.*pMgBpNa.*pNaBpCa - DK.*DMg.*pCaBpH.*pHBpNa.*pNaBpCa - DK.*DMg.*pCaBpNa.*pHBpCa.*pNaBpH - DH.*DMg.*pCaBpK.*pKBpNa.*pNaBpCa - DH.*DMg.*pCaBpNa.*pKBpCa.*pNaBpK - DK.*DNa.*pCaBpH.*pHBpMg.*pMgBpCa - DK.*DNa.*pCaBpMg.*pHBpCa.*pMgBpH - DH.*DNa.*pCaBpK.*pKBpMg.*pMgBpCa - DH.*DNa.*pCaBpMg.*pKBpCa.*pMgBpK - DCa.*DK.*pHBpNa.*pMgBpH.*pNaBpMg - DCa.*DK.*pHBpMg.*pMgBpNa.*pNaBpH - DCa.*DH.*pKBpNa.*pMgBpK.*pNaBpMg - DCa.*DH.*pKBpMg.*pMgBpNa.*pNaBpK - DMg.*DNa.*pCaBpH.*pHBpK.*pKBpCa - DMg.*DNa.*pCaBpK.*pHBpCa.*pKBpH - DCa.*DMg.*pHBpK.*pKBpNa.*pNaBpH - DCa.*DMg.*pHBpNa.*pKBpH.*pNaBpK - DCa.*DNa.*pHBpK.*pKBpMg.*pMgBpH - DCa.*DNa.*pHBpMg.*pKBpH.*pMgBpK - DK.*pCaBpH.*pHBpCa.*pMgBpNa.*pNaBpMg + DK.*pCaBpH.*pHBpNa.*pMgBpCa.*pNaBpMg + DK.*pCaBpH.*pHBpMg.*pMgBpNa.*pNaBpCa + DK.*pCaBpNa.*pHBpCa.*pMgBpH.*pNaBpMg - DK.*pCaBpNa.*pHBpMg.*pMgBpH.*pNaBpCa + DK.*pCaBpNa.*pHBpMg.*pMgBpCa.*pNaBpH + DK.*pCaBpMg.*pHBpCa.*pMgBpNa.*pNaBpH + DK.*pCaBpMg.*pHBpNa.*pMgBpH.*pNaBpCa - DK.*pCaBpMg.*pHBpNa.*pMgBpCa.*pNaBpH - DH.*pCaBpK.*pKBpCa.*pMgBpNa.*pNaBpMg + DH.*pCaBpK.*pKBpNa.*pMgBpCa.*pNaBpMg + DH.*pCaBpK.*pKBpMg.*pMgBpNa.*pNaBpCa + DH.*pCaBpNa.*pKBpCa.*pMgBpK.*pNaBpMg - DH.*pCaBpNa.*pKBpMg.*pMgBpK.*pNaBpCa + DH.*pCaBpNa.*pKBpMg.*pMgBpCa.*pNaBpK + DH.*pCaBpMg.*pKBpCa.*pMgBpNa.*pNaBpK + DH.*pCaBpMg.*pKBpNa.*pMgBpK.*pNaBpCa - DH.*pCaBpMg.*pKBpNa.*pMgBpCa.*pNaBpK + DMg.*pCaBpH.*pHBpK.*pKBpNa.*pNaBpCa - DMg.*pCaBpH.*pHBpCa.*pKBpNa.*pNaBpK + DMg.*pCaBpH.*pHBpNa.*pKBpCa.*pNaBpK + DMg.*pCaBpK.*pHBpCa.*pKBpNa.*pNaBpH + DMg.*pCaBpK.*pHBpNa.*pKBpH.*pNaBpCa - DMg.*pCaBpK.*pHBpNa.*pKBpCa.*pNaBpH - DMg.*pCaBpNa.*pHBpK.*pKBpH.*pNaBpCa + DMg.*pCaBpNa.*pHBpK.*pKBpCa.*pNaBpH + DMg.*pCaBpNa.*pHBpCa.*pKBpH.*pNaBpK + DNa.*pCaBpH.*pHBpK.*pKBpMg.*pMgBpCa - DNa.*pCaBpH.*pHBpCa.*pKBpMg.*pMgBpK + DNa.*pCaBpH.*pHBpMg.*pKBpCa.*pMgBpK + DNa.*pCaBpK.*pHBpCa.*pKBpMg.*pMgBpH + DNa.*pCaBpK.*pHBpMg.*pKBpH.*pMgBpCa - DNa.*pCaBpK.*pHBpMg.*pKBpCa.*pMgBpH - DNa.*pCaBpMg.*pHBpK.*pKBpH.*pMgBpCa + DNa.*pCaBpMg.*pHBpK.*pKBpCa.*pMgBpH + DNa.*pCaBpMg.*pHBpCa.*pKBpH.*pMgBpK - DCa.*pHBpK.*pKBpH.*pMgBpNa.*pNaBpMg + DCa.*pHBpK.*pKBpNa.*pMgBpH.*pNaBpMg + DCa.*pHBpK.*pKBpMg.*pMgBpNa.*pNaBpH + DCa.*pHBpNa.*pKBpH.*pMgBpK.*pNaBpMg + DCa.*pHBpNa.*pKBpMg.*pMgBpH.*pNaBpK - DCa.*pHBpNa.*pKBpMg.*pMgBpK.*pNaBpH + DCa.*pHBpMg.*pKBpH.*pMgBpNa.*pNaBpK - DCa.*pHBpMg.*pKBpNa.*pMgBpH.*pNaBpK + DCa.*pHBpMg.*pKBpNa.*pMgBpK.*pNaBpH + pCaBpH.*pHBpK.*pKBpCa.*pMgBpNa.*pNaBpMg - pCaBpH.*pHBpK.*pKBpNa.*pMgBpCa.*pNaBpMg - pCaBpH.*pHBpK.*pKBpMg.*pMgBpNa.*pNaBpCa + pCaBpH.*pHBpCa.*pKBpNa.*pMgBpK.*pNaBpMg + pCaBpH.*pHBpCa.*pKBpMg.*pMgBpNa.*pNaBpK - pCaBpH.*pHBpNa.*pKBpCa.*pMgBpK.*pNaBpMg + pCaBpH.*pHBpNa.*pKBpMg.*pMgBpK.*pNaBpCa - pCaBpH.*pHBpNa.*pKBpMg.*pMgBpCa.*pNaBpK - pCaBpH.*pHBpMg.*pKBpCa.*pMgBpNa.*pNaBpK - pCaBpH.*pHBpMg.*pKBpNa.*pMgBpK.*pNaBpCa + pCaBpH.*pHBpMg.*pKBpNa.*pMgBpCa.*pNaBpK + pCaBpK.*pHBpCa.*pKBpH.*pMgBpNa.*pNaBpMg - pCaBpK.*pHBpCa.*pKBpNa.*pMgBpH.*pNaBpMg - pCaBpK.*pHBpCa.*pKBpMg.*pMgBpNa.*pNaBpH - pCaBpK.*pHBpNa.*pKBpH.*pMgBpCa.*pNaBpMg + pCaBpK.*pHBpNa.*pKBpCa.*pMgBpH.*pNaBpMg - pCaBpK.*pHBpNa.*pKBpMg.*pMgBpH.*pNaBpCa + pCaBpK.*pHBpNa.*pKBpMg.*pMgBpCa.*pNaBpH - pCaBpK.*pHBpMg.*pKBpH.*pMgBpNa.*pNaBpCa + pCaBpK.*pHBpMg.*pKBpCa.*pMgBpNa.*pNaBpH + pCaBpK.*pHBpMg.*pKBpNa.*pMgBpH.*pNaBpCa - pCaBpK.*pHBpMg.*pKBpNa.*pMgBpCa.*pNaBpH + pCaBpNa.*pHBpK.*pKBpH.*pMgBpCa.*pNaBpMg - pCaBpNa.*pHBpK.*pKBpCa.*pMgBpH.*pNaBpMg + pCaBpNa.*pHBpK.*pKBpMg.*pMgBpH.*pNaBpCa - pCaBpNa.*pHBpK.*pKBpMg.*pMgBpCa.*pNaBpH - pCaBpNa.*pHBpCa.*pKBpH.*pMgBpK.*pNaBpMg - pCaBpNa.*pHBpCa.*pKBpMg.*pMgBpH.*pNaBpK + pCaBpNa.*pHBpCa.*pKBpMg.*pMgBpK.*pNaBpH + pCaBpNa.*pHBpMg.*pKBpH.*pMgBpK.*pNaBpCa - pCaBpNa.*pHBpMg.*pKBpH.*pMgBpCa.*pNaBpK + pCaBpNa.*pHBpMg.*pKBpCa.*pMgBpH.*pNaBpK - pCaBpNa.*pHBpMg.*pKBpCa.*pMgBpK.*pNaBpH + pCaBpMg.*pHBpK.*pKBpH.*pMgBpNa.*pNaBpCa - pCaBpMg.*pHBpK.*pKBpCa.*pMgBpNa.*pNaBpH - pCaBpMg.*pHBpK.*pKBpNa.*pMgBpH.*pNaBpCa + pCaBpMg.*pHBpK.*pKBpNa.*pMgBpCa.*pNaBpH - pCaBpMg.*pHBpCa.*pKBpH.*pMgBpNa.*pNaBpK + pCaBpMg.*pHBpCa.*pKBpNa.*pMgBpH.*pNaBpK - pCaBpMg.*pHBpCa.*pKBpNa.*pMgBpK.*pNaBpH - pCaBpMg.*pHBpNa.*pKBpH.*pMgBpK.*pNaBpCa + pCaBpMg.*pHBpNa.*pKBpH.*pMgBpCa.*pNaBpK - pCaBpMg.*pHBpNa.*pKBpCa.*pMgBpH.*pNaBpK + pCaBpMg.*pHBpNa.*pKBpCa.*pMgBpK.*pNaBpH - DCa.*DH.*DK.*DMg.*DNa);
dHdt=-(DK.*DMg.*PhiNa.*pCaBpNa.*pHBpCa + DCa.*DK.*PhiNa.*pHBpMg.*pMgBpNa + DCa.*DMg.*PhiNa.*pHBpK.*pKBpNa + DK.*DNa.*PhiMg.*pCaBpMg.*pHBpCa + DCa.*DK.*PhiMg.*pHBpNa.*pNaBpMg + DCa.*DNa.*PhiMg.*pHBpK.*pKBpMg + DMg.*DNa.*PhiK.*pCaBpK.*pHBpCa + DCa.*DMg.*PhiK.*pHBpNa.*pNaBpK + DCa.*DNa.*PhiK.*pHBpMg.*pMgBpK - DK.*DMg.*PhiH.*pCaBpNa.*pNaBpCa - DK.*DNa.*PhiH.*pCaBpMg.*pMgBpCa - DCa.*DK.*PhiH.*pMgBpNa.*pNaBpMg - DMg.*DNa.*PhiH.*pCaBpK.*pKBpCa - DCa.*DMg.*PhiH.*pKBpNa.*pNaBpK - DCa.*DNa.*PhiH.*pKBpMg.*pMgBpK + DK.*DMg.*PhiCa.*pHBpNa.*pNaBpCa + DK.*DNa.*PhiCa.*pHBpMg.*pMgBpCa + DMg.*DNa.*PhiCa.*pHBpK.*pKBpCa - DK.*PhiNa.*pCaBpNa.*pHBpMg.*pMgBpCa - DK.*PhiNa.*pCaBpMg.*pHBpCa.*pMgBpNa + DK.*PhiNa.*pCaBpMg.*pHBpNa.*pMgBpCa - DMg.*PhiNa.*pCaBpK.*pHBpCa.*pKBpNa + DMg.*PhiNa.*pCaBpK.*pHBpNa.*pKBpCa - DMg.*PhiNa.*pCaBpNa.*pHBpK.*pKBpCa - DCa.*PhiNa.*pHBpK.*pKBpMg.*pMgBpNa + DCa.*PhiNa.*pHBpNa.*pKBpMg.*pMgBpK - DCa.*PhiNa.*pHBpMg.*pKBpNa.*pMgBpK - DK.*PhiMg.*pCaBpNa.*pHBpCa.*pNaBpMg + DK.*PhiMg.*pCaBpNa.*pHBpMg.*pNaBpCa - DK.*PhiMg.*pCaBpMg.*pHBpNa.*pNaBpCa - DNa.*PhiMg.*pCaBpK.*pHBpCa.*pKBpMg + DNa.*PhiMg.*pCaBpK.*pHBpMg.*pKBpCa - DNa.*PhiMg.*pCaBpMg.*pHBpK.*pKBpCa - DCa.*PhiMg.*pHBpK.*pKBpNa.*pNaBpMg - DCa.*PhiMg.*pHBpNa.*pKBpMg.*pNaBpK + DCa.*PhiMg.*pHBpMg.*pKBpNa.*pNaBpK - DMg.*PhiK.*pCaBpK.*pHBpNa.*pNaBpCa + DMg.*PhiK.*pCaBpNa.*pHBpK.*pNaBpCa - DMg.*PhiK.*pCaBpNa.*pHBpCa.*pNaBpK - DNa.*PhiK.*pCaBpK.*pHBpMg.*pMgBpCa + DNa.*PhiK.*pCaBpMg.*pHBpK.*pMgBpCa - DNa.*PhiK.*pCaBpMg.*pHBpCa.*pMgBpK + DCa.*PhiK.*pHBpK.*pMgBpNa.*pNaBpMg - DCa.*PhiK.*pHBpNa.*pMgBpK.*pNaBpMg - DCa.*PhiK.*pHBpMg.*pMgBpNa.*pNaBpK + DK.*PhiH.*pCaBpNa.*pMgBpCa.*pNaBpMg + DK.*PhiH.*pCaBpMg.*pMgBpNa.*pNaBpCa + DMg.*PhiH.*pCaBpK.*pKBpNa.*pNaBpCa + DMg.*PhiH.*pCaBpNa.*pKBpCa.*pNaBpK + DNa.*PhiH.*pCaBpK.*pKBpMg.*pMgBpCa + DNa.*PhiH.*pCaBpMg.*pKBpCa.*pMgBpK + DCa.*PhiH.*pKBpNa.*pMgBpK.*pNaBpMg + DCa.*PhiH.*pKBpMg.*pMgBpNa.*pNaBpK + DK.*PhiCa.*pHBpCa.*pMgBpNa.*pNaBpMg - DK.*PhiCa.*pHBpNa.*pMgBpCa.*pNaBpMg - DK.*PhiCa.*pHBpMg.*pMgBpNa.*pNaBpCa - DMg.*PhiCa.*pHBpK.*pKBpNa.*pNaBpCa + DMg.*PhiCa.*pHBpCa.*pKBpNa.*pNaBpK - DMg.*PhiCa.*pHBpNa.*pKBpCa.*pNaBpK - DNa.*PhiCa.*pHBpK.*pKBpMg.*pMgBpCa + DNa.*PhiCa.*pHBpCa.*pKBpMg.*pMgBpK - DNa.*PhiCa.*pHBpMg.*pKBpCa.*pMgBpK + PhiNa.*pCaBpK.*pHBpCa.*pKBpMg.*pMgBpNa - PhiNa.*pCaBpK.*pHBpNa.*pKBpMg.*pMgBpCa - PhiNa.*pCaBpK.*pHBpMg.*pKBpCa.*pMgBpNa + PhiNa.*pCaBpK.*pHBpMg.*pKBpNa.*pMgBpCa + PhiNa.*pCaBpNa.*pHBpK.*pKBpMg.*pMgBpCa - PhiNa.*pCaBpNa.*pHBpCa.*pKBpMg.*pMgBpK + PhiNa.*pCaBpNa.*pHBpMg.*pKBpCa.*pMgBpK + PhiNa.*pCaBpMg.*pHBpK.*pKBpCa.*pMgBpNa - PhiNa.*pCaBpMg.*pHBpK.*pKBpNa.*pMgBpCa + PhiNa.*pCaBpMg.*pHBpCa.*pKBpNa.*pMgBpK - PhiNa.*pCaBpMg.*pHBpNa.*pKBpCa.*pMgBpK + PhiMg.*pCaBpK.*pHBpCa.*pKBpNa.*pNaBpMg - PhiMg.*pCaBpK.*pHBpNa.*pKBpCa.*pNaBpMg + PhiMg.*pCaBpK.*pHBpNa.*pKBpMg.*pNaBpCa - PhiMg.*pCaBpK.*pHBpMg.*pKBpNa.*pNaBpCa + PhiMg.*pCaBpNa.*pHBpK.*pKBpCa.*pNaBpMg - PhiMg.*pCaBpNa.*pHBpK.*pKBpMg.*pNaBpCa + PhiMg.*pCaBpNa.*pHBpCa.*pKBpMg.*pNaBpK - PhiMg.*pCaBpNa.*pHBpMg.*pKBpCa.*pNaBpK + PhiMg.*pCaBpMg.*pHBpK.*pKBpNa.*pNaBpCa - PhiMg.*pCaBpMg.*pHBpCa.*pKBpNa.*pNaBpK + PhiMg.*pCaBpMg.*pHBpNa.*pKBpCa.*pNaBpK - PhiK.*pCaBpK.*pHBpCa.*pMgBpNa.*pNaBpMg + PhiK.*pCaBpK.*pHBpNa.*pMgBpCa.*pNaBpMg + PhiK.*pCaBpK.*pHBpMg.*pMgBpNa.*pNaBpCa - PhiK.*pCaBpNa.*pHBpK.*pMgBpCa.*pNaBpMg + PhiK.*pCaBpNa.*pHBpCa.*pMgBpK.*pNaBpMg - PhiK.*pCaBpNa.*pHBpMg.*pMgBpK.*pNaBpCa + PhiK.*pCaBpNa.*pHBpMg.*pMgBpCa.*pNaBpK - PhiK.*pCaBpMg.*pHBpK.*pMgBpNa.*pNaBpCa + PhiK.*pCaBpMg.*pHBpCa.*pMgBpNa.*pNaBpK + PhiK.*pCaBpMg.*pHBpNa.*pMgBpK.*pNaBpCa - PhiK.*pCaBpMg.*pHBpNa.*pMgBpCa.*pNaBpK + PhiH.*pCaBpK.*pKBpCa.*pMgBpNa.*pNaBpMg - PhiH.*pCaBpK.*pKBpNa.*pMgBpCa.*pNaBpMg - PhiH.*pCaBpK.*pKBpMg.*pMgBpNa.*pNaBpCa - PhiH.*pCaBpNa.*pKBpCa.*pMgBpK.*pNaBpMg + PhiH.*pCaBpNa.*pKBpMg.*pMgBpK.*pNaBpCa - PhiH.*pCaBpNa.*pKBpMg.*pMgBpCa.*pNaBpK - PhiH.*pCaBpMg.*pKBpCa.*pMgBpNa.*pNaBpK - PhiH.*pCaBpMg.*pKBpNa.*pMgBpK.*pNaBpCa + PhiH.*pCaBpMg.*pKBpNa.*pMgBpCa.*pNaBpK - PhiCa.*pHBpK.*pKBpCa.*pMgBpNa.*pNaBpMg + PhiCa.*pHBpK.*pKBpNa.*pMgBpCa.*pNaBpMg + PhiCa.*pHBpK.*pKBpMg.*pMgBpNa.*pNaBpCa - PhiCa.*pHBpCa.*pKBpNa.*pMgBpK.*pNaBpMg - PhiCa.*pHBpCa.*pKBpMg.*pMgBpNa.*pNaBpK + PhiCa.*pHBpNa.*pKBpCa.*pMgBpK.*pNaBpMg - PhiCa.*pHBpNa.*pKBpMg.*pMgBpK.*pNaBpCa + PhiCa.*pHBpNa.*pKBpMg.*pMgBpCa.*pNaBpK + PhiCa.*pHBpMg.*pKBpCa.*pMgBpNa.*pNaBpK + PhiCa.*pHBpMg.*pKBpNa.*pMgBpK.*pNaBpCa - PhiCa.*pHBpMg.*pKBpNa.*pMgBpCa.*pNaBpK + DCa.*DK.*DMg.*DNa.*PhiH - DCa.*DK.*DMg.*PhiNa.*pHBpNa - DCa.*DK.*DNa.*PhiMg.*pHBpMg - DCa.*DMg.*DNa.*PhiK.*pHBpK - DK.*DMg.*DNa.*PhiCa.*pHBpCa)./D;
dNadt=(DH.*DK.*PhiNa.*pCaBpMg.*pMgBpCa + DK.*DMg.*PhiNa.*pCaBpH.*pHBpCa + DH.*DMg.*PhiNa.*pCaBpK.*pKBpCa + DCa.*DK.*PhiNa.*pHBpMg.*pMgBpH + DCa.*DH.*PhiNa.*pKBpMg.*pMgBpK + DCa.*DMg.*PhiNa.*pHBpK.*pKBpH - DH.*DK.*PhiMg.*pCaBpMg.*pNaBpCa - DCa.*DK.*PhiMg.*pHBpMg.*pNaBpH - DCa.*DH.*PhiMg.*pKBpMg.*pNaBpK - DH.*DMg.*PhiK.*pCaBpK.*pNaBpCa - DCa.*DH.*PhiK.*pMgBpK.*pNaBpMg - DCa.*DMg.*PhiK.*pHBpK.*pNaBpH - DK.*DMg.*PhiH.*pCaBpH.*pNaBpCa - DCa.*DK.*PhiH.*pMgBpH.*pNaBpMg - DCa.*DMg.*PhiH.*pKBpH.*pNaBpK - DH.*DK.*PhiCa.*pMgBpCa.*pNaBpMg - DK.*DMg.*PhiCa.*pHBpCa.*pNaBpH - DH.*DMg.*PhiCa.*pKBpCa.*pNaBpK - DK.*PhiNa.*pCaBpH.*pHBpMg.*pMgBpCa - DK.*PhiNa.*pCaBpMg.*pHBpCa.*pMgBpH - DH.*PhiNa.*pCaBpK.*pKBpMg.*pMgBpCa - DH.*PhiNa.*pCaBpMg.*pKBpCa.*pMgBpK - DMg.*PhiNa.*pCaBpH.*pHBpK.*pKBpCa - DMg.*PhiNa.*pCaBpK.*pHBpCa.*pKBpH - DCa.*PhiNa.*pHBpK.*pKBpMg.*pMgBpH - DCa.*PhiNa.*pHBpMg.*pKBpH.*pMgBpK - DK.*PhiMg.*pCaBpH.*pHBpCa.*pNaBpMg + DK.*PhiMg.*pCaBpH.*pHBpMg.*pNaBpCa + DK.*PhiMg.*pCaBpMg.*pHBpCa.*pNaBpH - DH.*PhiMg.*pCaBpK.*pKBpCa.*pNaBpMg + DH.*PhiMg.*pCaBpK.*pKBpMg.*pNaBpCa + DH.*PhiMg.*pCaBpMg.*pKBpCa.*pNaBpK - DCa.*PhiMg.*pHBpK.*pKBpH.*pNaBpMg + DCa.*PhiMg.*pHBpK.*pKBpMg.*pNaBpH + DCa.*PhiMg.*pHBpMg.*pKBpH.*pNaBpK + DH.*PhiK.*pCaBpK.*pMgBpCa.*pNaBpMg + DH.*PhiK.*pCaBpMg.*pMgBpK.*pNaBpCa - DH.*PhiK.*pCaBpMg.*pMgBpCa.*pNaBpK + DMg.*PhiK.*pCaBpH.*pHBpK.*pNaBpCa - DMg.*PhiK.*pCaBpH.*pHBpCa.*pNaBpK + DMg.*PhiK.*pCaBpK.*pHBpCa.*pNaBpH + DCa.*PhiK.*pHBpK.*pMgBpH.*pNaBpMg - DCa.*PhiK.*pHBpMg.*pMgBpH.*pNaBpK + DCa.*PhiK.*pHBpMg.*pMgBpK.*pNaBpH + DK.*PhiH.*pCaBpH.*pMgBpCa.*pNaBpMg + DK.*PhiH.*pCaBpMg.*pMgBpH.*pNaBpCa - DK.*PhiH.*pCaBpMg.*pMgBpCa.*pNaBpH + DMg.*PhiH.*pCaBpH.*pKBpCa.*pNaBpK + DMg.*PhiH.*pCaBpK.*pKBpH.*pNaBpCa - DMg.*PhiH.*pCaBpK.*pKBpCa.*pNaBpH + DCa.*PhiH.*pKBpH.*pMgBpK.*pNaBpMg + DCa.*PhiH.*pKBpMg.*pMgBpH.*pNaBpK - DCa.*PhiH.*pKBpMg.*pMgBpK.*pNaBpH + DK.*PhiCa.*pHBpCa.*pMgBpH.*pNaBpMg - DK.*PhiCa.*pHBpMg.*pMgBpH.*pNaBpCa + DK.*PhiCa.*pHBpMg.*pMgBpCa.*pNaBpH + DH.*PhiCa.*pKBpCa.*pMgBpK.*pNaBpMg - DH.*PhiCa.*pKBpMg.*pMgBpK.*pNaBpCa + DH.*PhiCa.*pKBpMg.*pMgBpCa.*pNaBpK - DMg.*PhiCa.*pHBpK.*pKBpH.*pNaBpCa + DMg.*PhiCa.*pHBpK.*pKBpCa.*pNaBpH + DMg.*PhiCa.*pHBpCa.*pKBpH.*pNaBpK + PhiNa.*pCaBpH.*pHBpK.*pKBpMg.*pMgBpCa - PhiNa.*pCaBpH.*pHBpCa.*pKBpMg.*pMgBpK + PhiNa.*pCaBpH.*pHBpMg.*pKBpCa.*pMgBpK + PhiNa.*pCaBpK.*pHBpCa.*pKBpMg.*pMgBpH + PhiNa.*pCaBpK.*pHBpMg.*pKBpH.*pMgBpCa - PhiNa.*pCaBpK.*pHBpMg.*pKBpCa.*pMgBpH - PhiNa.*pCaBpMg.*pHBpK.*pKBpH.*pMgBpCa + PhiNa.*pCaBpMg.*pHBpK.*pKBpCa.*pMgBpH + PhiNa.*pCaBpMg.*pHBpCa.*pKBpH.*pMgBpK + PhiMg.*pCaBpH.*pHBpK.*pKBpCa.*pNaBpMg - PhiMg.*pCaBpH.*pHBpK.*pKBpMg.*pNaBpCa + PhiMg.*pCaBpH.*pHBpCa.*pKBpMg.*pNaBpK - PhiMg.*pCaBpH.*pHBpMg.*pKBpCa.*pNaBpK + PhiMg.*pCaBpK.*pHBpCa.*pKBpH.*pNaBpMg - PhiMg.*pCaBpK.*pHBpCa.*pKBpMg.*pNaBpH - PhiMg.*pCaBpK.*pHBpMg.*pKBpH.*pNaBpCa + PhiMg.*pCaBpK.*pHBpMg.*pKBpCa.*pNaBpH + PhiMg.*pCaBpMg.*pHBpK.*pKBpH.*pNaBpCa - PhiMg.*pCaBpMg.*pHBpK.*pKBpCa.*pNaBpH - PhiMg.*pCaBpMg.*pHBpCa.*pKBpH.*pNaBpK - PhiK.*pCaBpH.*pHBpK.*pMgBpCa.*pNaBpMg + PhiK.*pCaBpH.*pHBpCa.*pMgBpK.*pNaBpMg - PhiK.*pCaBpH.*pHBpMg.*pMgBpK.*pNaBpCa + PhiK.*pCaBpH.*pHBpMg.*pMgBpCa.*pNaBpK - PhiK.*pCaBpK.*pHBpCa.*pMgBpH.*pNaBpMg + PhiK.*pCaBpK.*pHBpMg.*pMgBpH.*pNaBpCa - PhiK.*pCaBpK.*pHBpMg.*pMgBpCa.*pNaBpH - PhiK.*pCaBpMg.*pHBpK.*pMgBpH.*pNaBpCa + PhiK.*pCaBpMg.*pHBpK.*pMgBpCa.*pNaBpH + PhiK.*pCaBpMg.*pHBpCa.*pMgBpH.*pNaBpK - PhiK.*pCaBpMg.*pHBpCa.*pMgBpK.*pNaBpH - PhiH.*pCaBpH.*pKBpCa.*pMgBpK.*pNaBpMg + PhiH.*pCaBpH.*pKBpMg.*pMgBpK.*pNaBpCa - PhiH.*pCaBpH.*pKBpMg.*pMgBpCa.*pNaBpK - PhiH.*pCaBpK.*pKBpH.*pMgBpCa.*pNaBpMg + PhiH.*pCaBpK.*pKBpCa.*pMgBpH.*pNaBpMg - PhiH.*pCaBpK.*pKBpMg.*pMgBpH.*pNaBpCa + PhiH.*pCaBpK.*pKBpMg.*pMgBpCa.*pNaBpH - PhiH.*pCaBpMg.*pKBpH.*pMgBpK.*pNaBpCa + PhiH.*pCaBpMg.*pKBpH.*pMgBpCa.*pNaBpK - PhiH.*pCaBpMg.*pKBpCa.*pMgBpH.*pNaBpK + PhiH.*pCaBpMg.*pKBpCa.*pMgBpK.*pNaBpH + PhiCa.*pHBpK.*pKBpH.*pMgBpCa.*pNaBpMg - PhiCa.*pHBpK.*pKBpCa.*pMgBpH.*pNaBpMg + PhiCa.*pHBpK.*pKBpMg.*pMgBpH.*pNaBpCa - PhiCa.*pHBpK.*pKBpMg.*pMgBpCa.*pNaBpH - PhiCa.*pHBpCa.*pKBpH.*pMgBpK.*pNaBpMg - PhiCa.*pHBpCa.*pKBpMg.*pMgBpH.*pNaBpK + PhiCa.*pHBpCa.*pKBpMg.*pMgBpK.*pNaBpH + PhiCa.*pHBpMg.*pKBpH.*pMgBpK.*pNaBpCa - PhiCa.*pHBpMg.*pKBpH.*pMgBpCa.*pNaBpK + PhiCa.*pHBpMg.*pKBpCa.*pMgBpH.*pNaBpK - PhiCa.*pHBpMg.*pKBpCa.*pMgBpK.*pNaBpH - DCa.*DH.*DK.*DMg.*PhiNa + DCa.*DH.*DK.*PhiMg.*pNaBpMg + DCa.*DH.*DMg.*PhiK.*pNaBpK + DCa.*DK.*DMg.*PhiH.*pNaBpH + DH.*DK.*DMg.*PhiCa.*pNaBpCa)./D;
dKdt=-(DH.*DMg.*PhiNa.*pCaBpNa.*pKBpCa + DCa.*DH.*PhiNa.*pKBpMg.*pMgBpNa + DCa.*DMg.*PhiNa.*pHBpNa.*pKBpH + DH.*DNa.*PhiMg.*pCaBpMg.*pKBpCa + DCa.*DH.*PhiMg.*pKBpNa.*pNaBpMg + DCa.*DNa.*PhiMg.*pHBpMg.*pKBpH - DH.*DMg.*PhiK.*pCaBpNa.*pNaBpCa - DH.*DNa.*PhiK.*pCaBpMg.*pMgBpCa - DCa.*DH.*PhiK.*pMgBpNa.*pNaBpMg - DMg.*DNa.*PhiK.*pCaBpH.*pHBpCa - DCa.*DMg.*PhiK.*pHBpNa.*pNaBpH - DCa.*DNa.*PhiK.*pHBpMg.*pMgBpH + DMg.*DNa.*PhiH.*pCaBpH.*pKBpCa + DCa.*DMg.*PhiH.*pKBpNa.*pNaBpH + DCa.*DNa.*PhiH.*pKBpMg.*pMgBpH + DH.*DMg.*PhiCa.*pKBpNa.*pNaBpCa + DH.*DNa.*PhiCa.*pKBpMg.*pMgBpCa + DMg.*DNa.*PhiCa.*pHBpCa.*pKBpH - DH.*PhiNa.*pCaBpNa.*pKBpMg.*pMgBpCa - DH.*PhiNa.*pCaBpMg.*pKBpCa.*pMgBpNa + DH.*PhiNa.*pCaBpMg.*pKBpNa.*pMgBpCa + DMg.*PhiNa.*pCaBpH.*pHBpCa.*pKBpNa - DMg.*PhiNa.*pCaBpH.*pHBpNa.*pKBpCa - DMg.*PhiNa.*pCaBpNa.*pHBpCa.*pKBpH - DCa.*PhiNa.*pHBpNa.*pKBpMg.*pMgBpH - DCa.*PhiNa.*pHBpMg.*pKBpH.*pMgBpNa + DCa.*PhiNa.*pHBpMg.*pKBpNa.*pMgBpH - DH.*PhiMg.*pCaBpNa.*pKBpCa.*pNaBpMg + DH.*PhiMg.*pCaBpNa.*pKBpMg.*pNaBpCa - DH.*PhiMg.*pCaBpMg.*pKBpNa.*pNaBpCa + DNa.*PhiMg.*pCaBpH.*pHBpCa.*pKBpMg - DNa.*PhiMg.*pCaBpH.*pHBpMg.*pKBpCa - DNa.*PhiMg.*pCaBpMg.*pHBpCa.*pKBpH - DCa.*PhiMg.*pHBpNa.*pKBpH.*pNaBpMg + DCa.*PhiMg.*pHBpNa.*pKBpMg.*pNaBpH - DCa.*PhiMg.*pHBpMg.*pKBpNa.*pNaBpH + DH.*PhiK.*pCaBpNa.*pMgBpCa.*pNaBpMg + DH.*PhiK.*pCaBpMg.*pMgBpNa.*pNaBpCa + DMg.*PhiK.*pCaBpH.*pHBpNa.*pNaBpCa + DMg.*PhiK.*pCaBpNa.*pHBpCa.*pNaBpH + DNa.*PhiK.*pCaBpH.*pHBpMg.*pMgBpCa + DNa.*PhiK.*pCaBpMg.*pHBpCa.*pMgBpH + DCa.*PhiK.*pHBpNa.*pMgBpH.*pNaBpMg + DCa.*PhiK.*pHBpMg.*pMgBpNa.*pNaBpH - DMg.*PhiH.*pCaBpH.*pKBpNa.*pNaBpCa + DMg.*PhiH.*pCaBpNa.*pKBpH.*pNaBpCa - DMg.*PhiH.*pCaBpNa.*pKBpCa.*pNaBpH - DNa.*PhiH.*pCaBpH.*pKBpMg.*pMgBpCa + DNa.*PhiH.*pCaBpMg.*pKBpH.*pMgBpCa - DNa.*PhiH.*pCaBpMg.*pKBpCa.*pMgBpH + DCa.*PhiH.*pKBpH.*pMgBpNa.*pNaBpMg - DCa.*PhiH.*pKBpNa.*pMgBpH.*pNaBpMg - DCa.*PhiH.*pKBpMg.*pMgBpNa.*pNaBpH + DH.*PhiCa.*pKBpCa.*pMgBpNa.*pNaBpMg - DH.*PhiCa.*pKBpNa.*pMgBpCa.*pNaBpMg - DH.*PhiCa.*pKBpMg.*pMgBpNa.*pNaBpCa - DMg.*PhiCa.*pHBpCa.*pKBpNa.*pNaBpH - DMg.*PhiCa.*pHBpNa.*pKBpH.*pNaBpCa + DMg.*PhiCa.*pHBpNa.*pKBpCa.*pNaBpH - DNa.*PhiCa.*pHBpCa.*pKBpMg.*pMgBpH - DNa.*PhiCa.*pHBpMg.*pKBpH.*pMgBpCa + DNa.*PhiCa.*pHBpMg.*pKBpCa.*pMgBpH - PhiNa.*pCaBpH.*pHBpCa.*pKBpMg.*pMgBpNa + PhiNa.*pCaBpH.*pHBpNa.*pKBpMg.*pMgBpCa + PhiNa.*pCaBpH.*pHBpMg.*pKBpCa.*pMgBpNa - PhiNa.*pCaBpH.*pHBpMg.*pKBpNa.*pMgBpCa + PhiNa.*pCaBpNa.*pHBpCa.*pKBpMg.*pMgBpH + PhiNa.*pCaBpNa.*pHBpMg.*pKBpH.*pMgBpCa - PhiNa.*pCaBpNa.*pHBpMg.*pKBpCa.*pMgBpH + PhiNa.*pCaBpMg.*pHBpCa.*pKBpH.*pMgBpNa - PhiNa.*pCaBpMg.*pHBpCa.*pKBpNa.*pMgBpH - PhiNa.*pCaBpMg.*pHBpNa.*pKBpH.*pMgBpCa + PhiNa.*pCaBpMg.*pHBpNa.*pKBpCa.*pMgBpH - PhiMg.*pCaBpH.*pHBpCa.*pKBpNa.*pNaBpMg + PhiMg.*pCaBpH.*pHBpNa.*pKBpCa.*pNaBpMg - PhiMg.*pCaBpH.*pHBpNa.*pKBpMg.*pNaBpCa + PhiMg.*pCaBpH.*pHBpMg.*pKBpNa.*pNaBpCa + PhiMg.*pCaBpNa.*pHBpCa.*pKBpH.*pNaBpMg - PhiMg.*pCaBpNa.*pHBpCa.*pKBpMg.*pNaBpH - PhiMg.*pCaBpNa.*pHBpMg.*pKBpH.*pNaBpCa + PhiMg.*pCaBpNa.*pHBpMg.*pKBpCa.*pNaBpH + PhiMg.*pCaBpMg.*pHBpCa.*pKBpNa.*pNaBpH + PhiMg.*pCaBpMg.*pHBpNa.*pKBpH.*pNaBpCa - PhiMg.*pCaBpMg.*pHBpNa.*pKBpCa.*pNaBpH + PhiK.*pCaBpH.*pHBpCa.*pMgBpNa.*pNaBpMg - PhiK.*pCaBpH.*pHBpNa.*pMgBpCa.*pNaBpMg - PhiK.*pCaBpH.*pHBpMg.*pMgBpNa.*pNaBpCa - PhiK.*pCaBpNa.*pHBpCa.*pMgBpH.*pNaBpMg + PhiK.*pCaBpNa.*pHBpMg.*pMgBpH.*pNaBpCa - PhiK.*pCaBpNa.*pHBpMg.*pMgBpCa.*pNaBpH - PhiK.*pCaBpMg.*pHBpCa.*pMgBpNa.*pNaBpH - PhiK.*pCaBpMg.*pHBpNa.*pMgBpH.*pNaBpCa + PhiK.*pCaBpMg.*pHBpNa.*pMgBpCa.*pNaBpH - PhiH.*pCaBpH.*pKBpCa.*pMgBpNa.*pNaBpMg + PhiH.*pCaBpH.*pKBpNa.*pMgBpCa.*pNaBpMg + PhiH.*pCaBpH.*pKBpMg.*pMgBpNa.*pNaBpCa - PhiH.*pCaBpNa.*pKBpH.*pMgBpCa.*pNaBpMg + PhiH.*pCaBpNa.*pKBpCa.*pMgBpH.*pNaBpMg - PhiH.*pCaBpNa.*pKBpMg.*pMgBpH.*pNaBpCa + PhiH.*pCaBpNa.*pKBpMg.*pMgBpCa.*pNaBpH - PhiH.*pCaBpMg.*pKBpH.*pMgBpNa.*pNaBpCa + PhiH.*pCaBpMg.*pKBpCa.*pMgBpNa.*pNaBpH + PhiH.*pCaBpMg.*pKBpNa.*pMgBpH.*pNaBpCa - PhiH.*pCaBpMg.*pKBpNa.*pMgBpCa.*pNaBpH - PhiCa.*pHBpCa.*pKBpH.*pMgBpNa.*pNaBpMg + PhiCa.*pHBpCa.*pKBpNa.*pMgBpH.*pNaBpMg + PhiCa.*pHBpCa.*pKBpMg.*pMgBpNa.*pNaBpH + PhiCa.*pHBpNa.*pKBpH.*pMgBpCa.*pNaBpMg - PhiCa.*pHBpNa.*pKBpCa.*pMgBpH.*pNaBpMg + PhiCa.*pHBpNa.*pKBpMg.*pMgBpH.*pNaBpCa - PhiCa.*pHBpNa.*pKBpMg.*pMgBpCa.*pNaBpH + PhiCa.*pHBpMg.*pKBpH.*pMgBpNa.*pNaBpCa - PhiCa.*pHBpMg.*pKBpCa.*pMgBpNa.*pNaBpH - PhiCa.*pHBpMg.*pKBpNa.*pMgBpH.*pNaBpCa + PhiCa.*pHBpMg.*pKBpNa.*pMgBpCa.*pNaBpH + DCa.*DH.*DMg.*DNa.*PhiK - DCa.*DH.*DMg.*PhiNa.*pKBpNa - DCa.*DH.*DNa.*PhiMg.*pKBpMg - DCa.*DMg.*DNa.*PhiH.*pKBpH - DH.*DMg.*DNa.*PhiCa.*pKBpCa)./D;
dMgdt=-(DH.*DK.*PhiNa.*pCaBpNa.*pMgBpCa + DCa.*DK.*PhiNa.*pHBpNa.*pMgBpH + DCa.*DH.*PhiNa.*pKBpNa.*pMgBpK - DH.*DK.*PhiMg.*pCaBpNa.*pNaBpCa - DK.*DNa.*PhiMg.*pCaBpH.*pHBpCa - DH.*DNa.*PhiMg.*pCaBpK.*pKBpCa - DCa.*DK.*PhiMg.*pHBpNa.*pNaBpH - DCa.*DH.*PhiMg.*pKBpNa.*pNaBpK - DCa.*DNa.*PhiMg.*pHBpK.*pKBpH + DH.*DNa.*PhiK.*pCaBpK.*pMgBpCa + DCa.*DH.*PhiK.*pMgBpNa.*pNaBpK + DCa.*DNa.*PhiK.*pHBpK.*pMgBpH + DK.*DNa.*PhiH.*pCaBpH.*pMgBpCa + DCa.*DK.*PhiH.*pMgBpNa.*pNaBpH + DCa.*DNa.*PhiH.*pKBpH.*pMgBpK + DH.*DK.*PhiCa.*pMgBpNa.*pNaBpCa + DK.*DNa.*PhiCa.*pHBpCa.*pMgBpH + DH.*DNa.*PhiCa.*pKBpCa.*pMgBpK + DK.*PhiNa.*pCaBpH.*pHBpCa.*pMgBpNa - DK.*PhiNa.*pCaBpH.*pHBpNa.*pMgBpCa - DK.*PhiNa.*pCaBpNa.*pHBpCa.*pMgBpH + DH.*PhiNa.*pCaBpK.*pKBpCa.*pMgBpNa - DH.*PhiNa.*pCaBpK.*pKBpNa.*pMgBpCa - DH.*PhiNa.*pCaBpNa.*pKBpCa.*pMgBpK + DCa.*PhiNa.*pHBpK.*pKBpH.*pMgBpNa - DCa.*PhiNa.*pHBpK.*pKBpNa.*pMgBpH - DCa.*PhiNa.*pHBpNa.*pKBpH.*pMgBpK + DK.*PhiMg.*pCaBpH.*pHBpNa.*pNaBpCa + DK.*PhiMg.*pCaBpNa.*pHBpCa.*pNaBpH + DH.*PhiMg.*pCaBpK.*pKBpNa.*pNaBpCa + DH.*PhiMg.*pCaBpNa.*pKBpCa.*pNaBpK + DNa.*PhiMg.*pCaBpH.*pHBpK.*pKBpCa + DNa.*PhiMg.*pCaBpK.*pHBpCa.*pKBpH + DCa.*PhiMg.*pHBpK.*pKBpNa.*pNaBpH + DCa.*PhiMg.*pHBpNa.*pKBpH.*pNaBpK - DH.*PhiK.*pCaBpK.*pMgBpNa.*pNaBpCa + DH.*PhiK.*pCaBpNa.*pMgBpK.*pNaBpCa - DH.*PhiK.*pCaBpNa.*pMgBpCa.*pNaBpK - DNa.*PhiK.*pCaBpH.*pHBpK.*pMgBpCa + DNa.*PhiK.*pCaBpH.*pHBpCa.*pMgBpK - DNa.*PhiK.*pCaBpK.*pHBpCa.*pMgBpH - DCa.*PhiK.*pHBpK.*pMgBpNa.*pNaBpH - DCa.*PhiK.*pHBpNa.*pMgBpH.*pNaBpK + DCa.*PhiK.*pHBpNa.*pMgBpK.*pNaBpH - DK.*PhiH.*pCaBpH.*pMgBpNa.*pNaBpCa + DK.*PhiH.*pCaBpNa.*pMgBpH.*pNaBpCa - DK.*PhiH.*pCaBpNa.*pMgBpCa.*pNaBpH - DNa.*PhiH.*pCaBpH.*pKBpCa.*pMgBpK - DNa.*PhiH.*pCaBpK.*pKBpH.*pMgBpCa + DNa.*PhiH.*pCaBpK.*pKBpCa.*pMgBpH - DCa.*PhiH.*pKBpH.*pMgBpNa.*pNaBpK + DCa.*PhiH.*pKBpNa.*pMgBpH.*pNaBpK - DCa.*PhiH.*pKBpNa.*pMgBpK.*pNaBpH - DK.*PhiCa.*pHBpCa.*pMgBpNa.*pNaBpH - DK.*PhiCa.*pHBpNa.*pMgBpH.*pNaBpCa + DK.*PhiCa.*pHBpNa.*pMgBpCa.*pNaBpH - DH.*PhiCa.*pKBpCa.*pMgBpNa.*pNaBpK - DH.*PhiCa.*pKBpNa.*pMgBpK.*pNaBpCa + DH.*PhiCa.*pKBpNa.*pMgBpCa.*pNaBpK + DNa.*PhiCa.*pHBpK.*pKBpH.*pMgBpCa - DNa.*PhiCa.*pHBpK.*pKBpCa.*pMgBpH - DNa.*PhiCa.*pHBpCa.*pKBpH.*pMgBpK - PhiNa.*pCaBpH.*pHBpK.*pKBpCa.*pMgBpNa + PhiNa.*pCaBpH.*pHBpK.*pKBpNa.*pMgBpCa - PhiNa.*pCaBpH.*pHBpCa.*pKBpNa.*pMgBpK + PhiNa.*pCaBpH.*pHBpNa.*pKBpCa.*pMgBpK - PhiNa.*pCaBpK.*pHBpCa.*pKBpH.*pMgBpNa + PhiNa.*pCaBpK.*pHBpCa.*pKBpNa.*pMgBpH + PhiNa.*pCaBpK.*pHBpNa.*pKBpH.*pMgBpCa - PhiNa.*pCaBpK.*pHBpNa.*pKBpCa.*pMgBpH - PhiNa.*pCaBpNa.*pHBpK.*pKBpH.*pMgBpCa + PhiNa.*pCaBpNa.*pHBpK.*pKBpCa.*pMgBpH + PhiNa.*pCaBpNa.*pHBpCa.*pKBpH.*pMgBpK - PhiMg.*pCaBpH.*pHBpK.*pKBpNa.*pNaBpCa + PhiMg.*pCaBpH.*pHBpCa.*pKBpNa.*pNaBpK - PhiMg.*pCaBpH.*pHBpNa.*pKBpCa.*pNaBpK - PhiMg.*pCaBpK.*pHBpCa.*pKBpNa.*pNaBpH - PhiMg.*pCaBpK.*pHBpNa.*pKBpH.*pNaBpCa + PhiMg.*pCaBpK.*pHBpNa.*pKBpCa.*pNaBpH + PhiMg.*pCaBpNa.*pHBpK.*pKBpH.*pNaBpCa - PhiMg.*pCaBpNa.*pHBpK.*pKBpCa.*pNaBpH - PhiMg.*pCaBpNa.*pHBpCa.*pKBpH.*pNaBpK + PhiK.*pCaBpH.*pHBpK.*pMgBpNa.*pNaBpCa - PhiK.*pCaBpH.*pHBpCa.*pMgBpNa.*pNaBpK - PhiK.*pCaBpH.*pHBpNa.*pMgBpK.*pNaBpCa + PhiK.*pCaBpH.*pHBpNa.*pMgBpCa.*pNaBpK + PhiK.*pCaBpK.*pHBpCa.*pMgBpNa.*pNaBpH + PhiK.*pCaBpK.*pHBpNa.*pMgBpH.*pNaBpCa - PhiK.*pCaBpK.*pHBpNa.*pMgBpCa.*pNaBpH - PhiK.*pCaBpNa.*pHBpK.*pMgBpH.*pNaBpCa + PhiK.*pCaBpNa.*pHBpK.*pMgBpCa.*pNaBpH + PhiK.*pCaBpNa.*pHBpCa.*pMgBpH.*pNaBpK - PhiK.*pCaBpNa.*pHBpCa.*pMgBpK.*pNaBpH + PhiH.*pCaBpH.*pKBpCa.*pMgBpNa.*pNaBpK + PhiH.*pCaBpH.*pKBpNa.*pMgBpK.*pNaBpCa - PhiH.*pCaBpH.*pKBpNa.*pMgBpCa.*pNaBpK + PhiH.*pCaBpK.*pKBpH.*pMgBpNa.*pNaBpCa - PhiH.*pCaBpK.*pKBpCa.*pMgBpNa.*pNaBpH - PhiH.*pCaBpK.*pKBpNa.*pMgBpH.*pNaBpCa + PhiH.*pCaBpK.*pKBpNa.*pMgBpCa.*pNaBpH - PhiH.*pCaBpNa.*pKBpH.*pMgBpK.*pNaBpCa + PhiH.*pCaBpNa.*pKBpH.*pMgBpCa.*pNaBpK - PhiH.*pCaBpNa.*pKBpCa.*pMgBpH.*pNaBpK + PhiH.*pCaBpNa.*pKBpCa.*pMgBpK.*pNaBpH - PhiCa.*pHBpK.*pKBpH.*pMgBpNa.*pNaBpCa + PhiCa.*pHBpK.*pKBpCa.*pMgBpNa.*pNaBpH + PhiCa.*pHBpK.*pKBpNa.*pMgBpH.*pNaBpCa - PhiCa.*pHBpK.*pKBpNa.*pMgBpCa.*pNaBpH + PhiCa.*pHBpCa.*pKBpH.*pMgBpNa.*pNaBpK - PhiCa.*pHBpCa.*pKBpNa.*pMgBpH.*pNaBpK + PhiCa.*pHBpCa.*pKBpNa.*pMgBpK.*pNaBpH + PhiCa.*pHBpNa.*pKBpH.*pMgBpK.*pNaBpCa - PhiCa.*pHBpNa.*pKBpH.*pMgBpCa.*pNaBpK + PhiCa.*pHBpNa.*pKBpCa.*pMgBpH.*pNaBpK - PhiCa.*pHBpNa.*pKBpCa.*pMgBpK.*pNaBpH + DCa.*DH.*DK.*DNa.*PhiMg - DCa.*DH.*DK.*PhiNa.*pMgBpNa - DCa.*DH.*DNa.*PhiK.*pMgBpK - DCa.*DK.*DNa.*PhiH.*pMgBpH - DH.*DK.*DNa.*PhiCa.*pMgBpCa)./D;
dCadt=-(DH.*DK.*PhiNa.*pCaBpMg.*pMgBpNa + DK.*DMg.*PhiNa.*pCaBpH.*pHBpNa + DH.*DMg.*PhiNa.*pCaBpK.*pKBpNa + DH.*DK.*PhiMg.*pCaBpNa.*pNaBpMg + DK.*DNa.*PhiMg.*pCaBpH.*pHBpMg + DH.*DNa.*PhiMg.*pCaBpK.*pKBpMg + DH.*DMg.*PhiK.*pCaBpNa.*pNaBpK + DH.*DNa.*PhiK.*pCaBpMg.*pMgBpK + DMg.*DNa.*PhiK.*pCaBpH.*pHBpK + DK.*DMg.*PhiH.*pCaBpNa.*pNaBpH + DK.*DNa.*PhiH.*pCaBpMg.*pMgBpH + DMg.*DNa.*PhiH.*pCaBpK.*pKBpH - DH.*DK.*PhiCa.*pMgBpNa.*pNaBpMg - DK.*DMg.*PhiCa.*pHBpNa.*pNaBpH - DH.*DMg.*PhiCa.*pKBpNa.*pNaBpK - DK.*DNa.*PhiCa.*pHBpMg.*pMgBpH - DH.*DNa.*PhiCa.*pKBpMg.*pMgBpK - DMg.*DNa.*PhiCa.*pHBpK.*pKBpH - DK.*PhiNa.*pCaBpH.*pHBpMg.*pMgBpNa + DK.*PhiNa.*pCaBpNa.*pHBpMg.*pMgBpH - DK.*PhiNa.*pCaBpMg.*pHBpNa.*pMgBpH - DH.*PhiNa.*pCaBpK.*pKBpMg.*pMgBpNa + DH.*PhiNa.*pCaBpNa.*pKBpMg.*pMgBpK - DH.*PhiNa.*pCaBpMg.*pKBpNa.*pMgBpK - DMg.*PhiNa.*pCaBpH.*pHBpK.*pKBpNa - DMg.*PhiNa.*pCaBpK.*pHBpNa.*pKBpH + DMg.*PhiNa.*pCaBpNa.*pHBpK.*pKBpH - DK.*PhiMg.*pCaBpH.*pHBpNa.*pNaBpMg - DK.*PhiMg.*pCaBpNa.*pHBpMg.*pNaBpH + DK.*PhiMg.*pCaBpMg.*pHBpNa.*pNaBpH - DH.*PhiMg.*pCaBpK.*pKBpNa.*pNaBpMg - DH.*PhiMg.*pCaBpNa.*pKBpMg.*pNaBpK + DH.*PhiMg.*pCaBpMg.*pKBpNa.*pNaBpK - DNa.*PhiMg.*pCaBpH.*pHBpK.*pKBpMg - DNa.*PhiMg.*pCaBpK.*pHBpMg.*pKBpH + DNa.*PhiMg.*pCaBpMg.*pHBpK.*pKBpH + DH.*PhiK.*pCaBpK.*pMgBpNa.*pNaBpMg - DH.*PhiK.*pCaBpNa.*pMgBpK.*pNaBpMg - DH.*PhiK.*pCaBpMg.*pMgBpNa.*pNaBpK - DMg.*PhiK.*pCaBpH.*pHBpNa.*pNaBpK + DMg.*PhiK.*pCaBpK.*pHBpNa.*pNaBpH - DMg.*PhiK.*pCaBpNa.*pHBpK.*pNaBpH - DNa.*PhiK.*pCaBpH.*pHBpMg.*pMgBpK + DNa.*PhiK.*pCaBpK.*pHBpMg.*pMgBpH - DNa.*PhiK.*pCaBpMg.*pHBpK.*pMgBpH + DK.*PhiH.*pCaBpH.*pMgBpNa.*pNaBpMg - DK.*PhiH.*pCaBpNa.*pMgBpH.*pNaBpMg - DK.*PhiH.*pCaBpMg.*pMgBpNa.*pNaBpH + DMg.*PhiH.*pCaBpH.*pKBpNa.*pNaBpK - DMg.*PhiH.*pCaBpK.*pKBpNa.*pNaBpH - DMg.*PhiH.*pCaBpNa.*pKBpH.*pNaBpK + DNa.*PhiH.*pCaBpH.*pKBpMg.*pMgBpK - DNa.*PhiH.*pCaBpK.*pKBpMg.*pMgBpH - DNa.*PhiH.*pCaBpMg.*pKBpH.*pMgBpK + DK.*PhiCa.*pHBpNa.*pMgBpH.*pNaBpMg + DK.*PhiCa.*pHBpMg.*pMgBpNa.*pNaBpH + DH.*PhiCa.*pKBpNa.*pMgBpK.*pNaBpMg + DH.*PhiCa.*pKBpMg.*pMgBpNa.*pNaBpK + DMg.*PhiCa.*pHBpK.*pKBpNa.*pNaBpH + DMg.*PhiCa.*pHBpNa.*pKBpH.*pNaBpK + DNa.*PhiCa.*pHBpK.*pKBpMg.*pMgBpH + DNa.*PhiCa.*pHBpMg.*pKBpH.*pMgBpK + PhiNa.*pCaBpH.*pHBpK.*pKBpMg.*pMgBpNa - PhiNa.*pCaBpH.*pHBpNa.*pKBpMg.*pMgBpK + PhiNa.*pCaBpH.*pHBpMg.*pKBpNa.*pMgBpK + PhiNa.*pCaBpK.*pHBpNa.*pKBpMg.*pMgBpH + PhiNa.*pCaBpK.*pHBpMg.*pKBpH.*pMgBpNa - PhiNa.*pCaBpK.*pHBpMg.*pKBpNa.*pMgBpH - PhiNa.*pCaBpNa.*pHBpK.*pKBpMg.*pMgBpH - PhiNa.*pCaBpNa.*pHBpMg.*pKBpH.*pMgBpK - PhiNa.*pCaBpMg.*pHBpK.*pKBpH.*pMgBpNa + PhiNa.*pCaBpMg.*pHBpK.*pKBpNa.*pMgBpH + PhiNa.*pCaBpMg.*pHBpNa.*pKBpH.*pMgBpK + PhiMg.*pCaBpH.*pHBpK.*pKBpNa.*pNaBpMg + PhiMg.*pCaBpH.*pHBpNa.*pKBpMg.*pNaBpK - PhiMg.*pCaBpH.*pHBpMg.*pKBpNa.*pNaBpK + PhiMg.*pCaBpK.*pHBpNa.*pKBpH.*pNaBpMg - PhiMg.*pCaBpK.*pHBpNa.*pKBpMg.*pNaBpH + PhiMg.*pCaBpK.*pHBpMg.*pKBpNa.*pNaBpH - PhiMg.*pCaBpNa.*pHBpK.*pKBpH.*pNaBpMg + PhiMg.*pCaBpNa.*pHBpK.*pKBpMg.*pNaBpH + PhiMg.*pCaBpNa.*pHBpMg.*pKBpH.*pNaBpK - PhiMg.*pCaBpMg.*pHBpK.*pKBpNa.*pNaBpH - PhiMg.*pCaBpMg.*pHBpNa.*pKBpH.*pNaBpK - PhiK.*pCaBpH.*pHBpK.*pMgBpNa.*pNaBpMg + PhiK.*pCaBpH.*pHBpNa.*pMgBpK.*pNaBpMg + PhiK.*pCaBpH.*pHBpMg.*pMgBpNa.*pNaBpK - PhiK.*pCaBpK.*pHBpNa.*pMgBpH.*pNaBpMg - PhiK.*pCaBpK.*pHBpMg.*pMgBpNa.*pNaBpH + PhiK.*pCaBpNa.*pHBpK.*pMgBpH.*pNaBpMg - PhiK.*pCaBpNa.*pHBpMg.*pMgBpH.*pNaBpK + PhiK.*pCaBpNa.*pHBpMg.*pMgBpK.*pNaBpH + PhiK.*pCaBpMg.*pHBpK.*pMgBpNa.*pNaBpH + PhiK.*pCaBpMg.*pHBpNa.*pMgBpH.*pNaBpK - PhiK.*pCaBpMg.*pHBpNa.*pMgBpK.*pNaBpH - PhiH.*pCaBpH.*pKBpNa.*pMgBpK.*pNaBpMg - PhiH.*pCaBpH.*pKBpMg.*pMgBpNa.*pNaBpK - PhiH.*pCaBpK.*pKBpH.*pMgBpNa.*pNaBpMg + PhiH.*pCaBpK.*pKBpNa.*pMgBpH.*pNaBpMg + PhiH.*pCaBpK.*pKBpMg.*pMgBpNa.*pNaBpH + PhiH.*pCaBpNa.*pKBpH.*pMgBpK.*pNaBpMg + PhiH.*pCaBpNa.*pKBpMg.*pMgBpH.*pNaBpK - PhiH.*pCaBpNa.*pKBpMg.*pMgBpK.*pNaBpH + PhiH.*pCaBpMg.*pKBpH.*pMgBpNa.*pNaBpK - PhiH.*pCaBpMg.*pKBpNa.*pMgBpH.*pNaBpK + PhiH.*pCaBpMg.*pKBpNa.*pMgBpK.*pNaBpH + PhiCa.*pHBpK.*pKBpH.*pMgBpNa.*pNaBpMg - PhiCa.*pHBpK.*pKBpNa.*pMgBpH.*pNaBpMg - PhiCa.*pHBpK.*pKBpMg.*pMgBpNa.*pNaBpH - PhiCa.*pHBpNa.*pKBpH.*pMgBpK.*pNaBpMg - PhiCa.*pHBpNa.*pKBpMg.*pMgBpH.*pNaBpK + PhiCa.*pHBpNa.*pKBpMg.*pMgBpK.*pNaBpH - PhiCa.*pHBpMg.*pKBpH.*pMgBpNa.*pNaBpK + PhiCa.*pHBpMg.*pKBpNa.*pMgBpH.*pNaBpK - PhiCa.*pHBpMg.*pKBpNa.*pMgBpK.*pNaBpH + DH.*DK.*DMg.*DNa.*PhiCa - DH.*DK.*DMg.*PhiNa.*pCaBpNa - DH.*DK.*DNa.*PhiMg.*pCaBpMg - DH.*DMg.*DNa.*PhiK.*pCaBpK - DK.*DMg.*DNa.*PhiH.*pCaBpH)./D;

% rate units defined by Phi units
f(p.i.H_x,:) = dHdt;
f(p.i.Na_x,:) = dNadt;
f(p.i.K_x,:) = dKdt;
f(p.i.Mg_x,:) = dMgdt;
f(p.i.Ca_x,:) = dCadt;

end

%% Compute Extra-Matrix Ion Derivatives Using Rapid Equilibrium Approximation

function f = cCation_dXdt(x,f,p,J)
%% Compute Time Derivatives for Cations
H_c = x(p.i.H_c,:);     % M
K_c = x(p.i.K_c,:);     % M
Na_c = x(p.i.Na_c,:);   % M
Mg_c = x(p.i.Mg_c,:);   % M
Ca_c = x(p.i.Ca_c,:);   % M

%% Define Ligand Vector
Li = x(p.i.Lcyt,:); % M

%% Define Ligand Derivatives
dLidt = f(p.i.Lcyt,:); % M/s

%% Partial Derivatives of binding polynomials:
Pm_L = p.BP(p.i.Lcyt,:);
Pm_L2 = Pm_L.^2;
Pm_HBH = (1 + 1./p.KD(p.if.ligands,p.if.K)*K_c + 1./p.KD(p.if.ligands,p.if.Na)*Na_c ...
    + 1./p.KD(p.if.ligands,p.if.Mg)*Mg_c + 1./p.KD(p.if.ligands,p.if.Ca)*Ca_c);
Pm_KBK = (1 + 1./p.KD(p.if.ligands,p.if.H)*H_c + 1./p.KD(p.if.ligands,p.if.Na)*Na_c ...
    + 1./p.KD(p.if.ligands,p.if.Mg)*Mg_c + 1./p.KD(p.if.ligands,p.if.Ca)*Ca_c);
Pm_NBN = (1 + 1./p.KD(p.if.ligands,p.if.H)*H_c + 1./p.KD(p.if.ligands,p.if.K)*K_c ...
    + 1./p.KD(p.if.ligands,p.if.Mg)*Mg_c + 1./p.KD(p.if.ligands,p.if.Ca)*Ca_c);
Pm_MBM = (1 + 1./p.KD(p.if.ligands,p.if.H)*H_c + 1./p.KD(p.if.ligands,p.if.K)*K_c ...
    + 1./p.KD(p.if.ligands,p.if.Na)*Na_c + 1./p.KD(p.if.ligands,p.if.Ca)*Ca_c);
Pm_CBC = (1 + 1./p.KD(p.if.ligands,p.if.H)*H_c + 1./p.KD(p.if.ligands,p.if.K)*K_c ...
    + 1./p.KD(p.if.ligands,p.if.Na)*Na_c + 1./p.KD(p.if.ligands,p.if.Mg)*Mg_c);

pHBpH = + sum(Pm_HBH.*Li./Pm_L2./(p.KD(p.if.ligands,p.if.H)*ones(1,size(Li,2))),1);
pHBpK = - H_c.*sum(Li./Pm_L2 ...
    ./(p.KD(p.if.ligands,p.if.H).*p.KD(p.if.ligands,p.if.K)*ones(1,size(Li,2))),1);
pHBpNa = - H_c.*sum(Li./Pm_L2 ...
    ./(p.KD(p.if.ligands,p.if.H).*p.KD(p.if.ligands,p.if.Na)*ones(1,size(Li,2))),1);
pHBpMg = - H_c.*sum(Li./Pm_L2 ...
    ./(p.KD(p.if.ligands,p.if.H).*p.KD(p.if.ligands,p.if.Mg)*ones(1,size(Li,2))),1);
pHBpCa = - H_c.*sum(Li./Pm_L2 ...
    ./(p.KD(p.if.ligands,p.if.H).*p.KD(p.if.ligands,p.if.Ca)*ones(1,size(Li,2))),1);

pKBpH = - K_c.*sum(Li./Pm_L2 ...
    ./(p.KD(p.if.ligands,p.if.K).*p.KD(p.if.ligands,p.if.H)*ones(1,size(Li,2))),1);
pKBpK = + sum(Pm_KBK.*Li./Pm_L2./(p.KD(p.if.ligands,p.if.K)*ones(1,size(Li,2))),1);
pKBpNa = - K_c.*sum(Li./Pm_L2 ...
    ./(p.KD(p.if.ligands,p.if.K).*p.KD(p.if.ligands,p.if.Na)*ones(1,size(Li,2))),1);
pKBpMg = - K_c.*sum(Li./Pm_L2 ...
    ./(p.KD(p.if.ligands,p.if.K).*p.KD(p.if.ligands,p.if.Mg)*ones(1,size(Li,2))),1);
pKBpCa = - K_c.*sum(Li./Pm_L2 ...
    ./(p.KD(p.if.ligands,p.if.K).*p.KD(p.if.ligands,p.if.Ca)*ones(1,size(Li,2))),1);

pNaBpH = - Na_c.*sum(Li./Pm_L2 ...
    ./(p.KD(p.if.ligands,p.if.Na).*p.KD(p.if.ligands,p.if.H)*ones(1,size(Li,2))),1);
pNaBpK = - Na_c.*sum(Li./Pm_L2 ...
    ./(p.KD(p.if.ligands,p.if.Na).*p.KD(p.if.ligands,p.if.K)*ones(1,size(Li,2))),1);
pNaBpNa = + sum(Pm_NBN.*Li./Pm_L2./(p.KD(p.if.ligands,p.if.Na)*ones(1,size(Li,2))),1);
pNaBpMg = - Na_c.*sum(Li./Pm_L2 ...
    ./(p.KD(p.if.ligands,p.if.Na).*p.KD(p.if.ligands,p.if.Mg)*ones(1,size(Li,2))),1);
pNaBpCa = - Na_c.*sum(Li./Pm_L2 ...
    ./(p.KD(p.if.ligands,p.if.Na).*p.KD(p.if.ligands,p.if.Ca)*ones(1,size(Li,2))),1);

pMgBpH = - Mg_c.*sum(Li./Pm_L2 ...
    ./(p.KD(p.if.ligands,p.if.Mg).*p.KD(p.if.ligands,p.if.H)*ones(1,size(Li,2))),1);
pMgBpK = - Mg_c.*sum(Li./Pm_L2 ...
    ./(p.KD(p.if.ligands,p.if.Mg).*p.KD(p.if.ligands,p.if.K)*ones(1,size(Li,2))),1);
pMgBpNa = - Mg_c.*sum(Li./Pm_L2 ...
    ./(p.KD(p.if.ligands,p.if.Mg).*p.KD(p.if.ligands,p.if.Na)*ones(1,size(Li,2))),1);
pMgBpMg = + sum(Pm_MBM.*Li./Pm_L2./(p.KD(p.if.ligands,p.if.Mg)*ones(1,size(Li,2))),1);
pMgBpCa = - Mg_c.*sum(Li./Pm_L2 ...
    ./(p.KD(p.if.ligands,p.if.Mg).*p.KD(p.if.ligands,p.if.Ca)*ones(1,size(Li,2))),1);

pCaBpH = - Ca_c.*sum(Li./Pm_L2 ...
    ./(p.KD(p.if.ligands,p.if.Ca).*p.KD(p.if.ligands,p.if.H)*ones(1,size(Li,2))),1);
pCaBpK = - Ca_c.*sum(Li./Pm_L2 ...
    ./(p.KD(p.if.ligands,p.if.Ca).*p.KD(p.if.ligands,p.if.K)*ones(1,size(Li,2))),1);
pCaBpNa = - Ca_c.*sum(Li./Pm_L2 ...
    ./(p.KD(p.if.ligands,p.if.Ca).*p.KD(p.if.ligands,p.if.Na)*ones(1,size(Li,2))),1);
pCaBpMg = - Ca_c.*sum(Li./Pm_L2 ...
    ./(p.KD(p.if.ligands,p.if.Ca).*p.KD(p.if.ligands,p.if.Mg)*ones(1,size(Li,2))),1);
pCaBpCa = + sum(Pm_CBC.*Li./Pm_L2 ...
    ./(p.KD(p.if.ligands,p.if.Ca)*ones(1,size(Li,2))),1);

%% PHI's:
PhiH = - H_c.*sum(dLidt./Pm_L ...
    ./(p.KD(p.if.ligands,p.if.H)*ones(1,size(Li,2)))) ...
    - ( - 5*J.CI.JNADH - 1*J.CIII.Jc3 - 4*J.CIV ...
    + (p.F1FO.nH-1)*J.F1FO + 2*J.PiC + J.mHleak ...
    + 1*J.mKHE - 0*J.CI.JH2O2 + J.FCCP + J.mNHE + J.FCCP2 + 2*J.mCHE2 + 3*J.mCHE3...
    - J.Inject(p.if.H,:))/p.Vol.H2O_cyt; % add CHE
PhiK = - K_c.*sum(dLidt./Pm_L ...
    ./(p.KD(p.if.ligands,p.if.K)*ones(1,size(Li,2)))) ...
    - (J.mKleak - J.mKHE - J.Inject(p.if.K,:))/p.Vol.H2O_cyt;
PhiNa = - Na_c.*sum(dLidt./Pm_L ...
    ./(p.KD(p.if.ligands,p.if.Na)*ones(1,size(Li,2))))...
    - (3*J.NCLX - J.mNHE - J.Inject(p.if.Na,:))/p.Vol.H2O_cyt;
PhiMg = - Mg_c.*sum(dLidt./Pm_L ...
    ./(p.KD(p.if.ligands,p.if.Mg)*ones(1,size(Li,2)))) ...
    + J.Inject(p.if.Mg,:)/p.Vol.H2O_cyt;
PhiCa = - Ca_c.*sum(dLidt./Pm_L ...
    ./(p.KD(p.if.ligands,p.if.Ca)*ones(1,size(Li,2)))) ...
    - (J.MCU - J.NCLX - J.mCHE2 - J.mCHE3 + J.RaM - J.Inject(p.if.Ca,:))/p.Vol.H2O_cyt; % add CHE

%% additional buffers for [H+], [K+], [Na+], [Mg2+] and [Ca2+]
BH = p.cBuff.Htot;      % M, eg HEPES(pKa=7.55), Tris(pKa=8.1), MOPS(pKa=7.2)
K_BH = p.cBuff.KD_H; % M, sets buffer pH

% extra buffers added to cytosol
EDTA = p.cBuff.EDTA;   % M, total
EGTA = p.cBuff.EGTA;   % M, total

% EDTA binding poly (must include 2nd protonation state)
P_EDTA = 1 + H_c./p.EB.KD(p.if2.EDTA,p.if2.H) ...
    + H_c.^2./p.EB.KD(p.if2.EDTA,p.if2.H)./p.EB.KD(p.if2.EDTA,p.if2.HH)...
    + Na_c./p.EB.KD(p.if2.EDTA,p.if2.Na) ...
    + K_c./p.EB.KD(p.if2.EDTA,p.if2.K) ...
    + Mg_c./p.EB.KD(p.if2.EDTA,p.if2.Mg) ...
    + Ca_c./p.EB.KD(p.if2.EDTA,p.if2.Ca);

% EGTA binding poly (must include 2nd protonation state)
P_EGTA = 1 + H_c./p.EB.KD(p.if2.EGTA,p.if2.H) ...
    + H_c.^2./p.EB.KD(p.if2.EGTA,p.if2.H)./p.EB.KD(p.if2.EGTA,p.if2.HH)...
    + Na_c./p.EB.KD(p.if2.EGTA,p.if2.Na) ...
    + K_c./p.EB.KD(p.if2.EGTA,p.if2.K) ...
    + Mg_c./p.EB.KD(p.if2.EGTA,p.if2.Mg) ...
    + Ca_c./p.EB.KD(p.if2.EGTA,p.if2.Ca);

% K and Na binding to OM proteins/lipids negligible
BK=0;K_BK=inf;BNa=0;K_BNa=inf;

% Currently, do not allow Mg/Ca to bind to OM proteins/lipids
BMg=0;K_BMg=inf;%BCa=0;K_BCa=inf;

% BSA and CaGreen buffering
BCa = p.cBuff.Ca.Btot;
K_BCa = p.cBuff.Ca.K;
n_BCa = p.cBuff.Ca.n;

% Compute partial derivatives of EGTA and EDTA species
% here HEGTA is taken to be HEGTA+H2EGTA
% and HEDTA = HEDTA + H2EDTA
pHEGTApH = EGTA/p.EB.KD(p.if2.EGTA,p.if2.H)./P_EGTA.^2.*((P_EGTA ...
    - H_c/p.EB.KD(p.if2.EGTA,p.if2.H) ...
    - 2*H_c.^2/p.EB.KD(p.if2.EGTA,p.if2.H)/p.EB.KD(p.if2.EGTA,p.if2.HH))...
    + H_c/p.EB.KD(p.if2.EGTA,p.if2.HH).*(2*P_EGTA ...
    - H_c/p.EB.KD(p.if2.EGTA,p.if2.H) ...
    - H_c.^2/p.EB.KD(p.if2.EGTA,p.if2.H)/p.EB.KD(p.if2.EGTA,p.if2.HH)));
pHEGTApNa = -H_c.*EGTA/p.EB.KD(p.if2.EGTA,p.if2.H) ...
    /p.EB.KD(p.if2.EGTA,p.if2.Na)./P_EGTA.^2.*(1 ...
    + H_c/p.EB.KD(p.if2.EGTA,p.if2.HH));
pHEGTApK = -H_c.*EGTA/p.EB.KD(p.if2.EGTA,p.if2.H) ...
    /p.EB.KD(p.if2.EGTA,p.if2.K)./P_EGTA.^2.*(1 ...
    + H_c/p.EB.KD(p.if2.EGTA,p.if2.HH));
pHEGTApMg = -H_c.*EGTA/p.EB.KD(p.if2.EGTA,p.if2.H) ...
    /p.EB.KD(p.if2.EGTA,p.if2.Mg)./P_EGTA.^2.*(1 ...
    + H_c/p.EB.KD(p.if2.EGTA,p.if2.HH));
pHEGTApCa = -H_c.*EGTA/p.EB.KD(p.if2.EGTA,p.if2.H) ...
    /p.EB.KD(p.if2.EGTA,p.if2.Ca)./P_EGTA.^2.*(1 ...
    + H_c/p.EB.KD(p.if2.EGTA,p.if2.HH));

pNaEGTApH = -Na_c.*EGTA/p.EB.KD(p.if2.EGTA,p.if2.Na)./P_EGTA.^2 ...
    .*(1/p.EB.KD(p.if2.EGTA,p.if2.H) ...
    + 2*H_c/p.EB.KD(p.if2.EGTA,p.if2.H)/p.EB.KD(p.if2.EGTA,p.if2.HH));
pNaEGTApNa = EGTA/p.EB.KD(p.if2.EGTA,p.if2.Na)./P_EGTA.^2.*(P_EGTA ...
    - Na_c/p.EB.KD(p.if2.EGTA,p.if2.Na));
pNaEGTApK = -Na_c.*EGTA/p.EB.KD(p.if2.EGTA,p.if2.Na)./P_EGTA.^2 ...
    /p.EB.KD(p.if2.EGTA,p.if2.K);
pNaEGTApMg = -Na_c.*EGTA/p.EB.KD(p.if2.EGTA,p.if2.Na)./P_EGTA.^2 ...
    /p.EB.KD(p.if2.EGTA,p.if2.Mg);
pNaEGTApCa = -Na_c.*EGTA/p.EB.KD(p.if2.EGTA,p.if2.Na)./P_EGTA.^2 ...
    /p.EB.KD(p.if2.EGTA,p.if2.Ca);

pKEGTApH = -K_c.*EGTA/p.EB.KD(p.if2.EGTA,p.if2.K)./P_EGTA.^2 ...
    .*(1/p.EB.KD(p.if2.EGTA,p.if2.H) ...
    + 2*H_c/p.EB.KD(p.if2.EGTA,p.if2.H)/p.EB.KD(p.if2.EGTA,p.if2.HH));
pKEGTApNa = -K_c.*EGTA/p.EB.KD(p.if2.EGTA,p.if2.K)./P_EGTA.^2 ...
    /p.EB.KD(p.if2.EGTA,p.if2.Na);
pKEGTApK = EGTA/p.EB.KD(p.if2.EGTA,p.if2.K)./P_EGTA.^2.*(P_EGTA ...
    - K_c/p.EB.KD(p.if2.EGTA,p.if2.K));
pKEGTApMg = -K_c.*EGTA/p.EB.KD(p.if2.EGTA,p.if2.K)./P_EGTA.^2 ...
    /p.EB.KD(p.if2.EGTA,p.if2.Mg);
pKEGTApCa = -K_c.*EGTA/p.EB.KD(p.if2.EGTA,p.if2.K)./P_EGTA.^2 ...
    /p.EB.KD(p.if2.EGTA,p.if2.Ca);

pMgEGTApH = -Mg_c.*EGTA/p.EB.KD(p.if2.EGTA,p.if2.Mg)./P_EGTA.^2 ...
    .*(1/p.EB.KD(p.if2.EGTA,p.if2.H) ...
    + 2*H_c/p.EB.KD(p.if2.EGTA,p.if2.H)/p.EB.KD(p.if2.EGTA,p.if2.HH));
pMgEGTApNa = -Mg_c.*EGTA/p.EB.KD(p.if2.EGTA,p.if2.Mg)./P_EGTA.^2 ...
    /p.EB.KD(p.if2.EGTA,p.if2.Na);
pMgEGTApK = -Mg_c.*EGTA/p.EB.KD(p.if2.EGTA,p.if2.Mg)./P_EGTA.^2 ...
    /p.EB.KD(p.if2.EGTA,p.if2.K);
pMgEGTApMg = EGTA/p.EB.KD(p.if2.EGTA,p.if2.Mg)./P_EGTA.^2.*(P_EGTA ...
    - Mg_c/p.EB.KD(p.if2.EGTA,p.if2.Mg));
pMgEGTApCa = -Mg_c.*EGTA/p.EB.KD(p.if2.EGTA,p.if2.Mg)./P_EGTA.^2 ...
    /p.EB.KD(p.if2.EGTA,p.if2.Ca);

pCaEGTApH = -Ca_c.*EGTA/p.EB.KD(p.if2.EGTA,p.if2.Ca)./P_EGTA.^2 ...
    .*(1/p.EB.KD(p.if2.EGTA,p.if2.H) ...
    + 2*H_c/p.EB.KD(p.if2.EGTA,p.if2.H)/p.EB.KD(p.if2.EGTA,p.if2.HH));
pCaEGTApNa = -Ca_c.*EGTA/p.EB.KD(p.if2.EGTA,p.if2.Ca)./P_EGTA.^2 ...
    /p.EB.KD(p.if2.EGTA,p.if2.Na);
pCaEGTApK = -Ca_c.*EGTA/p.EB.KD(p.if2.EGTA,p.if2.Ca)./P_EGTA.^2 ...
    /p.EB.KD(p.if2.EGTA,p.if2.K);
pCaEGTApMg = -Ca_c.*EGTA/p.EB.KD(p.if2.EGTA,p.if2.Ca)./P_EGTA.^2 ...
    /p.EB.KD(p.if2.EGTA,p.if2.Mg);
pCaEGTApCa = EGTA/p.EB.KD(p.if2.EGTA,p.if2.Ca)./P_EGTA.^2.*(P_EGTA ...
    - Ca_c/p.EB.KD(p.if2.EGTA,p.if2.Ca));

pHEDTApH = EDTA/p.EB.KD(p.if2.EDTA,p.if2.H)./P_EDTA.^2.*((P_EDTA ...
    - H_c/p.EB.KD(p.if2.EDTA,p.if2.H) ...
    - 2*H_c.^2/p.EB.KD(p.if2.EDTA,p.if2.H)/p.EB.KD(p.if2.EDTA,p.if2.HH))...
    + H_c/p.EB.KD(p.if2.EDTA,p.if2.HH).*(2*P_EDTA ...
    - H_c/p.EB.KD(p.if2.EDTA,p.if2.H) ...
    - H_c.^2/p.EB.KD(p.if2.EDTA,p.if2.H)/p.EB.KD(p.if2.EDTA,p.if2.HH)));
pHEDTApNa = -H_c.*EDTA/p.EB.KD(p.if2.EDTA,p.if2.H) ...
    /p.EB.KD(p.if2.EDTA,p.if2.Na)./P_EDTA.^2.*(1 ...
    + H_c/p.EB.KD(p.if2.EDTA,p.if2.HH));
pHEDTApK = -H_c.*EDTA/p.EB.KD(p.if2.EDTA,p.if2.H) ...
    /p.EB.KD(p.if2.EDTA,p.if2.K)./P_EDTA.^2.*(1 ...
    + H_c/p.EB.KD(p.if2.EDTA,p.if2.HH));
pHEDTApMg = -H_c.*EDTA/p.EB.KD(p.if2.EDTA,p.if2.H) ...
    /p.EB.KD(p.if2.EDTA,p.if2.Mg)./P_EDTA.^2.*(1 ...
    + H_c/p.EB.KD(p.if2.EDTA,p.if2.HH));
pHEDTApCa = -H_c.*EDTA/p.EB.KD(p.if2.EDTA,p.if2.H) ...
    /p.EB.KD(p.if2.EDTA,p.if2.Ca)./P_EDTA.^2.*(1 ...
    + H_c/p.EB.KD(p.if2.EDTA,p.if2.HH));

pNaEDTApH = -Na_c.*EDTA/p.EB.KD(p.if2.EDTA,p.if2.Na)./P_EDTA.^2 ...
    .*(1/p.EB.KD(p.if2.EDTA,p.if2.H) ...
    + 2*H_c/p.EB.KD(p.if2.EDTA,p.if2.H)/p.EB.KD(p.if2.EDTA,p.if2.HH));
pNaEDTApNa = EDTA/p.EB.KD(p.if2.EDTA,p.if2.Na)./P_EDTA.^2.*(P_EDTA ...
    - Na_c/p.EB.KD(p.if2.EDTA,p.if2.Na));
pNaEDTApK = -Na_c.*EDTA/p.EB.KD(p.if2.EDTA,p.if2.Na)./P_EDTA.^2 ...
    /p.EB.KD(p.if2.EDTA,p.if2.K);
pNaEDTApMg = -Na_c.*EDTA/p.EB.KD(p.if2.EDTA,p.if2.Na)./P_EDTA.^2 ...
    /p.EB.KD(p.if2.EDTA,p.if2.Mg);
pNaEDTApCa = -Na_c.*EDTA/p.EB.KD(p.if2.EDTA,p.if2.Na)./P_EDTA.^2 ...
    /p.EB.KD(p.if2.EDTA,p.if2.Ca);

pKEDTApH = -K_c.*EDTA/p.EB.KD(p.if2.EDTA,p.if2.K)./P_EDTA.^2 ...
    .*(1/p.EB.KD(p.if2.EDTA,p.if2.H) ...
    + 2*H_c/p.EB.KD(p.if2.EDTA,p.if2.H)/p.EB.KD(p.if2.EDTA,p.if2.HH));
pKEDTApNa = -K_c.*EDTA/p.EB.KD(p.if2.EDTA,p.if2.K)./P_EDTA.^2 ...
    /p.EB.KD(p.if2.EDTA,p.if2.Na);
pKEDTApK = EDTA/p.EB.KD(p.if2.EDTA,p.if2.K)./P_EDTA.^2.*(P_EDTA ...
    - K_c/p.EB.KD(p.if2.EDTA,p.if2.K));
pKEDTApMg = -K_c.*EDTA/p.EB.KD(p.if2.EDTA,p.if2.K)./P_EDTA.^2 ...
    /p.EB.KD(p.if2.EDTA,p.if2.Mg);
pKEDTApCa = -K_c.*EDTA/p.EB.KD(p.if2.EDTA,p.if2.K)./P_EDTA.^2 ...
    /p.EB.KD(p.if2.EDTA,p.if2.Ca);

pMgEDTApH = -Mg_c.*EDTA/p.EB.KD(p.if2.EDTA,p.if2.Mg)./P_EDTA.^2 ...
    .*(1/p.EB.KD(p.if2.EDTA,p.if2.H) ...
    + 2*H_c/p.EB.KD(p.if2.EDTA,p.if2.H)/p.EB.KD(p.if2.EDTA,p.if2.HH));
pMgEDTApNa = -Mg_c.*EDTA/p.EB.KD(p.if2.EDTA,p.if2.Mg)./P_EDTA.^2 ...
    /p.EB.KD(p.if2.EDTA,p.if2.Na);
pMgEDTApK = -Mg_c.*EDTA/p.EB.KD(p.if2.EDTA,p.if2.Mg)./P_EDTA.^2 ...
    /p.EB.KD(p.if2.EDTA,p.if2.K);
pMgEDTApMg = EDTA/p.EB.KD(p.if2.EDTA,p.if2.Mg)./P_EDTA.^2.*(P_EDTA ...
    - Mg_c/p.EB.KD(p.if2.EDTA,p.if2.Mg));
pMgEDTApCa = -Mg_c.*EDTA/p.EB.KD(p.if2.EDTA,p.if2.Mg)./P_EDTA.^2 ...
    /p.EB.KD(p.if2.EDTA,p.if2.Ca);

pCaEDTApH = -Ca_c.*EDTA/p.EB.KD(p.if2.EDTA,p.if2.Ca)./P_EDTA.^2 ...
    .*(1/p.EB.KD(p.if2.EDTA,p.if2.H) ...
    + 2*H_c/p.EB.KD(p.if2.EDTA,p.if2.H)/p.EB.KD(p.if2.EDTA,p.if2.HH));
pCaEDTApNa = -Ca_c.*EDTA/p.EB.KD(p.if2.EDTA,p.if2.Ca)./P_EDTA.^2 ...
    /p.EB.KD(p.if2.EDTA,p.if2.Na);
pCaEDTApK = -Ca_c.*EDTA/p.EB.KD(p.if2.EDTA,p.if2.Ca)./P_EDTA.^2 ...
    /p.EB.KD(p.if2.EDTA,p.if2.K);
pCaEDTApMg = -Ca_c.*EDTA/p.EB.KD(p.if2.EDTA,p.if2.Ca)./P_EDTA.^2 ...
    /p.EB.KD(p.if2.EDTA,p.if2.Mg);
pCaEDTApCa = EDTA/p.EB.KD(p.if2.EDTA,p.if2.Ca)./P_EDTA.^2.*(P_EDTA ...
    - Ca_c/p.EB.KD(p.if2.EDTA,p.if2.Ca));

% add to partial derivatives already computed
pHBpH = pHBpH + pHEGTApH + pHEDTApH;
pHBpNa = pHBpNa + pHEGTApNa + pHEDTApNa;
pHBpK = pHBpK + pHEGTApK + pHEDTApK;
pHBpMg = pHBpMg + pHEGTApMg + pHEDTApMg;
pHBpCa = pHBpCa + pHEGTApCa + pHEDTApCa;

pKBpH = pKBpH + pKEGTApH + pKEDTApH;
pKBpK = pKBpK + pKEGTApK + pKEDTApK;
pKBpNa = pKBpNa + pKEGTApNa + pKEDTApNa;
pKBpMg = pKBpMg + pKEGTApMg + pKEDTApMg;
pKBpCa = pKBpCa + pKEGTApCa + pKEDTApCa;

pNaBpH = pNaBpH + pNaEGTApH + pNaEDTApH;
pNaBpK = pNaBpK  + pNaEGTApK + pNaEDTApK;
pNaBpNa = pNaBpNa + pNaEGTApNa + pNaEDTApNa;
pNaBpMg = pNaBpMg + pNaEGTApMg + pNaEDTApMg;
pNaBpCa = pNaBpCa + pNaEGTApCa + pNaEDTApCa;

pMgBpH = pMgBpH + pMgEGTApH + pMgEDTApH;
pMgBpK = pMgBpK + pMgEGTApK + pMgEDTApK;
pMgBpNa = pMgBpNa + pMgEGTApNa + pMgEDTApNa;
pMgBpMg = pMgBpMg + pMgEGTApMg + pMgEDTApMg;
pMgBpCa = pMgBpCa + pMgEGTApCa + pMgEDTApCa;

pCaBpH = pCaBpH + pCaEGTApH + pCaEDTApH;
pCaBpK = pCaBpK + pCaEGTApK + pCaEDTApK;
pCaBpNa = pCaBpNa + pCaEGTApNa + pCaEDTApNa;
pCaBpMg = pCaBpMg + pCaEGTApMg + pCaEDTApMg;
pCaBpCa = pCaBpCa + pCaEGTApCa + pCaEDTApCa;

%% Buffering Terms:
DH = 1 + pHBpH + BH./K_BH./(1+H_c./K_BH).^2;
DK = 1 + pKBpK + BK./K_BK./(1+K_c./K_BK).^2;
DNa = 1 + pNaBpNa + BNa./K_BNa./(1+Na_c./K_BNa).^2;
DMg = 1 + pMgBpMg + BMg./K_BMg./(1+Mg_c./K_BMg).^2;
DCa = 1 + pCaBpCa + n_BCa^2*Ca_c.^(n_BCa-1).*BCa/K_BCa^n_BCa./(1+(Ca_c/K_BCa).^n_BCa).^2;

%% "simplified" solution to linear eqns. from Vinnakota et al., 2008
dHdt = -(DK.*DMg.*PhiNa.*pCaBpNa.*pHBpCa + DCa.*DK.*PhiNa.*pHBpMg.*pMgBpNa + DCa.*DMg.*PhiNa.*pHBpK.*pKBpNa + DK.*DNa.*PhiMg.*pCaBpMg.*pHBpCa + DCa.*DK.*PhiMg.*pHBpNa.*pNaBpMg + DCa.*DNa.*PhiMg.*pHBpK.*pKBpMg + DMg.*DNa.*PhiK.*pCaBpK.*pHBpCa + DCa.*DMg.*PhiK.*pHBpNa.*pNaBpK + DCa.*DNa.*PhiK.*pHBpMg.*pMgBpK - DK.*DMg.*PhiH.*pCaBpNa.*pNaBpCa - DK.*DNa.*PhiH.*pCaBpMg.*pMgBpCa - DCa.*DK.*PhiH.*pMgBpNa.*pNaBpMg - DMg.*DNa.*PhiH.*pCaBpK.*pKBpCa - DCa.*DMg.*PhiH.*pKBpNa.*pNaBpK - DCa.*DNa.*PhiH.*pKBpMg.*pMgBpK + DK.*DMg.*PhiCa.*pHBpNa.*pNaBpCa + DK.*DNa.*PhiCa.*pHBpMg.*pMgBpCa + DMg.*DNa.*PhiCa.*pHBpK.*pKBpCa - DK.*PhiNa.*pCaBpNa.*pHBpMg.*pMgBpCa - DK.*PhiNa.*pCaBpMg.*pHBpCa.*pMgBpNa + DK.*PhiNa.*pCaBpMg.*pHBpNa.*pMgBpCa - DMg.*PhiNa.*pCaBpK.*pHBpCa.*pKBpNa + DMg.*PhiNa.*pCaBpK.*pHBpNa.*pKBpCa - DMg.*PhiNa.*pCaBpNa.*pHBpK.*pKBpCa - DCa.*PhiNa.*pHBpK.*pKBpMg.*pMgBpNa + DCa.*PhiNa.*pHBpNa.*pKBpMg.*pMgBpK - DCa.*PhiNa.*pHBpMg.*pKBpNa.*pMgBpK - DK.*PhiMg.*pCaBpNa.*pHBpCa.*pNaBpMg + DK.*PhiMg.*pCaBpNa.*pHBpMg.*pNaBpCa - DK.*PhiMg.*pCaBpMg.*pHBpNa.*pNaBpCa - DNa.*PhiMg.*pCaBpK.*pHBpCa.*pKBpMg + DNa.*PhiMg.*pCaBpK.*pHBpMg.*pKBpCa - DNa.*PhiMg.*pCaBpMg.*pHBpK.*pKBpCa - DCa.*PhiMg.*pHBpK.*pKBpNa.*pNaBpMg - DCa.*PhiMg.*pHBpNa.*pKBpMg.*pNaBpK + DCa.*PhiMg.*pHBpMg.*pKBpNa.*pNaBpK - DMg.*PhiK.*pCaBpK.*pHBpNa.*pNaBpCa + DMg.*PhiK.*pCaBpNa.*pHBpK.*pNaBpCa - DMg.*PhiK.*pCaBpNa.*pHBpCa.*pNaBpK - DNa.*PhiK.*pCaBpK.*pHBpMg.*pMgBpCa + DNa.*PhiK.*pCaBpMg.*pHBpK.*pMgBpCa - DNa.*PhiK.*pCaBpMg.*pHBpCa.*pMgBpK + DCa.*PhiK.*pHBpK.*pMgBpNa.*pNaBpMg - DCa.*PhiK.*pHBpNa.*pMgBpK.*pNaBpMg - DCa.*PhiK.*pHBpMg.*pMgBpNa.*pNaBpK + DK.*PhiH.*pCaBpNa.*pMgBpCa.*pNaBpMg + DK.*PhiH.*pCaBpMg.*pMgBpNa.*pNaBpCa + DMg.*PhiH.*pCaBpK.*pKBpNa.*pNaBpCa + DMg.*PhiH.*pCaBpNa.*pKBpCa.*pNaBpK + DNa.*PhiH.*pCaBpK.*pKBpMg.*pMgBpCa + DNa.*PhiH.*pCaBpMg.*pKBpCa.*pMgBpK + DCa.*PhiH.*pKBpNa.*pMgBpK.*pNaBpMg + DCa.*PhiH.*pKBpMg.*pMgBpNa.*pNaBpK + DK.*PhiCa.*pHBpCa.*pMgBpNa.*pNaBpMg - DK.*PhiCa.*pHBpNa.*pMgBpCa.*pNaBpMg - DK.*PhiCa.*pHBpMg.*pMgBpNa.*pNaBpCa - DMg.*PhiCa.*pHBpK.*pKBpNa.*pNaBpCa + DMg.*PhiCa.*pHBpCa.*pKBpNa.*pNaBpK - DMg.*PhiCa.*pHBpNa.*pKBpCa.*pNaBpK - DNa.*PhiCa.*pHBpK.*pKBpMg.*pMgBpCa + DNa.*PhiCa.*pHBpCa.*pKBpMg.*pMgBpK - DNa.*PhiCa.*pHBpMg.*pKBpCa.*pMgBpK + PhiNa.*pCaBpK.*pHBpCa.*pKBpMg.*pMgBpNa - PhiNa.*pCaBpK.*pHBpNa.*pKBpMg.*pMgBpCa - PhiNa.*pCaBpK.*pHBpMg.*pKBpCa.*pMgBpNa + PhiNa.*pCaBpK.*pHBpMg.*pKBpNa.*pMgBpCa + PhiNa.*pCaBpNa.*pHBpK.*pKBpMg.*pMgBpCa - PhiNa.*pCaBpNa.*pHBpCa.*pKBpMg.*pMgBpK + PhiNa.*pCaBpNa.*pHBpMg.*pKBpCa.*pMgBpK + PhiNa.*pCaBpMg.*pHBpK.*pKBpCa.*pMgBpNa - PhiNa.*pCaBpMg.*pHBpK.*pKBpNa.*pMgBpCa + PhiNa.*pCaBpMg.*pHBpCa.*pKBpNa.*pMgBpK - PhiNa.*pCaBpMg.*pHBpNa.*pKBpCa.*pMgBpK + PhiMg.*pCaBpK.*pHBpCa.*pKBpNa.*pNaBpMg - PhiMg.*pCaBpK.*pHBpNa.*pKBpCa.*pNaBpMg + PhiMg.*pCaBpK.*pHBpNa.*pKBpMg.*pNaBpCa - PhiMg.*pCaBpK.*pHBpMg.*pKBpNa.*pNaBpCa + PhiMg.*pCaBpNa.*pHBpK.*pKBpCa.*pNaBpMg - PhiMg.*pCaBpNa.*pHBpK.*pKBpMg.*pNaBpCa + PhiMg.*pCaBpNa.*pHBpCa.*pKBpMg.*pNaBpK - PhiMg.*pCaBpNa.*pHBpMg.*pKBpCa.*pNaBpK + PhiMg.*pCaBpMg.*pHBpK.*pKBpNa.*pNaBpCa - PhiMg.*pCaBpMg.*pHBpCa.*pKBpNa.*pNaBpK + PhiMg.*pCaBpMg.*pHBpNa.*pKBpCa.*pNaBpK - PhiK.*pCaBpK.*pHBpCa.*pMgBpNa.*pNaBpMg + PhiK.*pCaBpK.*pHBpNa.*pMgBpCa.*pNaBpMg + PhiK.*pCaBpK.*pHBpMg.*pMgBpNa.*pNaBpCa - PhiK.*pCaBpNa.*pHBpK.*pMgBpCa.*pNaBpMg + PhiK.*pCaBpNa.*pHBpCa.*pMgBpK.*pNaBpMg - PhiK.*pCaBpNa.*pHBpMg.*pMgBpK.*pNaBpCa + PhiK.*pCaBpNa.*pHBpMg.*pMgBpCa.*pNaBpK - PhiK.*pCaBpMg.*pHBpK.*pMgBpNa.*pNaBpCa + PhiK.*pCaBpMg.*pHBpCa.*pMgBpNa.*pNaBpK + PhiK.*pCaBpMg.*pHBpNa.*pMgBpK.*pNaBpCa - PhiK.*pCaBpMg.*pHBpNa.*pMgBpCa.*pNaBpK + PhiH.*pCaBpK.*pKBpCa.*pMgBpNa.*pNaBpMg - PhiH.*pCaBpK.*pKBpNa.*pMgBpCa.*pNaBpMg - PhiH.*pCaBpK.*pKBpMg.*pMgBpNa.*pNaBpCa - PhiH.*pCaBpNa.*pKBpCa.*pMgBpK.*pNaBpMg + PhiH.*pCaBpNa.*pKBpMg.*pMgBpK.*pNaBpCa - PhiH.*pCaBpNa.*pKBpMg.*pMgBpCa.*pNaBpK - PhiH.*pCaBpMg.*pKBpCa.*pMgBpNa.*pNaBpK - PhiH.*pCaBpMg.*pKBpNa.*pMgBpK.*pNaBpCa + PhiH.*pCaBpMg.*pKBpNa.*pMgBpCa.*pNaBpK - PhiCa.*pHBpK.*pKBpCa.*pMgBpNa.*pNaBpMg + PhiCa.*pHBpK.*pKBpNa.*pMgBpCa.*pNaBpMg + PhiCa.*pHBpK.*pKBpMg.*pMgBpNa.*pNaBpCa - PhiCa.*pHBpCa.*pKBpNa.*pMgBpK.*pNaBpMg - PhiCa.*pHBpCa.*pKBpMg.*pMgBpNa.*pNaBpK + PhiCa.*pHBpNa.*pKBpCa.*pMgBpK.*pNaBpMg - PhiCa.*pHBpNa.*pKBpMg.*pMgBpK.*pNaBpCa + PhiCa.*pHBpNa.*pKBpMg.*pMgBpCa.*pNaBpK + PhiCa.*pHBpMg.*pKBpCa.*pMgBpNa.*pNaBpK + PhiCa.*pHBpMg.*pKBpNa.*pMgBpK.*pNaBpCa - PhiCa.*pHBpMg.*pKBpNa.*pMgBpCa.*pNaBpK + DCa.*DK.*DMg.*DNa.*PhiH - DCa.*DK.*DMg.*PhiNa.*pHBpNa - DCa.*DK.*DNa.*PhiMg.*pHBpMg - DCa.*DMg.*DNa.*PhiK.*pHBpK - DK.*DMg.*DNa.*PhiCa.*pHBpCa)./(DH.*DK.*DMg.*pCaBpNa.*pNaBpCa + DH.*DK.*DNa.*pCaBpMg.*pMgBpCa + DCa.*DH.*DK.*pMgBpNa.*pNaBpMg + DK.*DMg.*DNa.*pCaBpH.*pHBpCa + DH.*DMg.*DNa.*pCaBpK.*pKBpCa + DCa.*DK.*DMg.*pHBpNa.*pNaBpH + DCa.*DH.*DMg.*pKBpNa.*pNaBpK + DCa.*DK.*DNa.*pHBpMg.*pMgBpH + DCa.*DH.*DNa.*pKBpMg.*pMgBpK + DCa.*DMg.*DNa.*pHBpK.*pKBpH - DH.*DK.*pCaBpNa.*pMgBpCa.*pNaBpMg - DH.*DK.*pCaBpMg.*pMgBpNa.*pNaBpCa - DK.*DMg.*pCaBpH.*pHBpNa.*pNaBpCa - DK.*DMg.*pCaBpNa.*pHBpCa.*pNaBpH - DH.*DMg.*pCaBpK.*pKBpNa.*pNaBpCa - DH.*DMg.*pCaBpNa.*pKBpCa.*pNaBpK - DK.*DNa.*pCaBpH.*pHBpMg.*pMgBpCa - DK.*DNa.*pCaBpMg.*pHBpCa.*pMgBpH - DH.*DNa.*pCaBpK.*pKBpMg.*pMgBpCa - DH.*DNa.*pCaBpMg.*pKBpCa.*pMgBpK - DCa.*DK.*pHBpNa.*pMgBpH.*pNaBpMg - DCa.*DK.*pHBpMg.*pMgBpNa.*pNaBpH - DCa.*DH.*pKBpNa.*pMgBpK.*pNaBpMg - DCa.*DH.*pKBpMg.*pMgBpNa.*pNaBpK - DMg.*DNa.*pCaBpH.*pHBpK.*pKBpCa - DMg.*DNa.*pCaBpK.*pHBpCa.*pKBpH - DCa.*DMg.*pHBpK.*pKBpNa.*pNaBpH - DCa.*DMg.*pHBpNa.*pKBpH.*pNaBpK - DCa.*DNa.*pHBpK.*pKBpMg.*pMgBpH - DCa.*DNa.*pHBpMg.*pKBpH.*pMgBpK - DK.*pCaBpH.*pHBpCa.*pMgBpNa.*pNaBpMg + DK.*pCaBpH.*pHBpNa.*pMgBpCa.*pNaBpMg + DK.*pCaBpH.*pHBpMg.*pMgBpNa.*pNaBpCa + DK.*pCaBpNa.*pHBpCa.*pMgBpH.*pNaBpMg - DK.*pCaBpNa.*pHBpMg.*pMgBpH.*pNaBpCa + DK.*pCaBpNa.*pHBpMg.*pMgBpCa.*pNaBpH + DK.*pCaBpMg.*pHBpCa.*pMgBpNa.*pNaBpH + DK.*pCaBpMg.*pHBpNa.*pMgBpH.*pNaBpCa - DK.*pCaBpMg.*pHBpNa.*pMgBpCa.*pNaBpH - DH.*pCaBpK.*pKBpCa.*pMgBpNa.*pNaBpMg + DH.*pCaBpK.*pKBpNa.*pMgBpCa.*pNaBpMg + DH.*pCaBpK.*pKBpMg.*pMgBpNa.*pNaBpCa + DH.*pCaBpNa.*pKBpCa.*pMgBpK.*pNaBpMg - DH.*pCaBpNa.*pKBpMg.*pMgBpK.*pNaBpCa + DH.*pCaBpNa.*pKBpMg.*pMgBpCa.*pNaBpK + DH.*pCaBpMg.*pKBpCa.*pMgBpNa.*pNaBpK + DH.*pCaBpMg.*pKBpNa.*pMgBpK.*pNaBpCa - DH.*pCaBpMg.*pKBpNa.*pMgBpCa.*pNaBpK + DMg.*pCaBpH.*pHBpK.*pKBpNa.*pNaBpCa - DMg.*pCaBpH.*pHBpCa.*pKBpNa.*pNaBpK + DMg.*pCaBpH.*pHBpNa.*pKBpCa.*pNaBpK + DMg.*pCaBpK.*pHBpCa.*pKBpNa.*pNaBpH + DMg.*pCaBpK.*pHBpNa.*pKBpH.*pNaBpCa - DMg.*pCaBpK.*pHBpNa.*pKBpCa.*pNaBpH - DMg.*pCaBpNa.*pHBpK.*pKBpH.*pNaBpCa + DMg.*pCaBpNa.*pHBpK.*pKBpCa.*pNaBpH + DMg.*pCaBpNa.*pHBpCa.*pKBpH.*pNaBpK + DNa.*pCaBpH.*pHBpK.*pKBpMg.*pMgBpCa - DNa.*pCaBpH.*pHBpCa.*pKBpMg.*pMgBpK + DNa.*pCaBpH.*pHBpMg.*pKBpCa.*pMgBpK + DNa.*pCaBpK.*pHBpCa.*pKBpMg.*pMgBpH + DNa.*pCaBpK.*pHBpMg.*pKBpH.*pMgBpCa - DNa.*pCaBpK.*pHBpMg.*pKBpCa.*pMgBpH - DNa.*pCaBpMg.*pHBpK.*pKBpH.*pMgBpCa + DNa.*pCaBpMg.*pHBpK.*pKBpCa.*pMgBpH + DNa.*pCaBpMg.*pHBpCa.*pKBpH.*pMgBpK - DCa.*pHBpK.*pKBpH.*pMgBpNa.*pNaBpMg + DCa.*pHBpK.*pKBpNa.*pMgBpH.*pNaBpMg + DCa.*pHBpK.*pKBpMg.*pMgBpNa.*pNaBpH + DCa.*pHBpNa.*pKBpH.*pMgBpK.*pNaBpMg + DCa.*pHBpNa.*pKBpMg.*pMgBpH.*pNaBpK - DCa.*pHBpNa.*pKBpMg.*pMgBpK.*pNaBpH + DCa.*pHBpMg.*pKBpH.*pMgBpNa.*pNaBpK - DCa.*pHBpMg.*pKBpNa.*pMgBpH.*pNaBpK + DCa.*pHBpMg.*pKBpNa.*pMgBpK.*pNaBpH + pCaBpH.*pHBpK.*pKBpCa.*pMgBpNa.*pNaBpMg - pCaBpH.*pHBpK.*pKBpNa.*pMgBpCa.*pNaBpMg - pCaBpH.*pHBpK.*pKBpMg.*pMgBpNa.*pNaBpCa + pCaBpH.*pHBpCa.*pKBpNa.*pMgBpK.*pNaBpMg + pCaBpH.*pHBpCa.*pKBpMg.*pMgBpNa.*pNaBpK - pCaBpH.*pHBpNa.*pKBpCa.*pMgBpK.*pNaBpMg + pCaBpH.*pHBpNa.*pKBpMg.*pMgBpK.*pNaBpCa - pCaBpH.*pHBpNa.*pKBpMg.*pMgBpCa.*pNaBpK - pCaBpH.*pHBpMg.*pKBpCa.*pMgBpNa.*pNaBpK - pCaBpH.*pHBpMg.*pKBpNa.*pMgBpK.*pNaBpCa + pCaBpH.*pHBpMg.*pKBpNa.*pMgBpCa.*pNaBpK + pCaBpK.*pHBpCa.*pKBpH.*pMgBpNa.*pNaBpMg - pCaBpK.*pHBpCa.*pKBpNa.*pMgBpH.*pNaBpMg - pCaBpK.*pHBpCa.*pKBpMg.*pMgBpNa.*pNaBpH - pCaBpK.*pHBpNa.*pKBpH.*pMgBpCa.*pNaBpMg + pCaBpK.*pHBpNa.*pKBpCa.*pMgBpH.*pNaBpMg - pCaBpK.*pHBpNa.*pKBpMg.*pMgBpH.*pNaBpCa + pCaBpK.*pHBpNa.*pKBpMg.*pMgBpCa.*pNaBpH - pCaBpK.*pHBpMg.*pKBpH.*pMgBpNa.*pNaBpCa + pCaBpK.*pHBpMg.*pKBpCa.*pMgBpNa.*pNaBpH + pCaBpK.*pHBpMg.*pKBpNa.*pMgBpH.*pNaBpCa - pCaBpK.*pHBpMg.*pKBpNa.*pMgBpCa.*pNaBpH + pCaBpNa.*pHBpK.*pKBpH.*pMgBpCa.*pNaBpMg - pCaBpNa.*pHBpK.*pKBpCa.*pMgBpH.*pNaBpMg + pCaBpNa.*pHBpK.*pKBpMg.*pMgBpH.*pNaBpCa - pCaBpNa.*pHBpK.*pKBpMg.*pMgBpCa.*pNaBpH - pCaBpNa.*pHBpCa.*pKBpH.*pMgBpK.*pNaBpMg - pCaBpNa.*pHBpCa.*pKBpMg.*pMgBpH.*pNaBpK + pCaBpNa.*pHBpCa.*pKBpMg.*pMgBpK.*pNaBpH + pCaBpNa.*pHBpMg.*pKBpH.*pMgBpK.*pNaBpCa - pCaBpNa.*pHBpMg.*pKBpH.*pMgBpCa.*pNaBpK + pCaBpNa.*pHBpMg.*pKBpCa.*pMgBpH.*pNaBpK - pCaBpNa.*pHBpMg.*pKBpCa.*pMgBpK.*pNaBpH + pCaBpMg.*pHBpK.*pKBpH.*pMgBpNa.*pNaBpCa - pCaBpMg.*pHBpK.*pKBpCa.*pMgBpNa.*pNaBpH - pCaBpMg.*pHBpK.*pKBpNa.*pMgBpH.*pNaBpCa + pCaBpMg.*pHBpK.*pKBpNa.*pMgBpCa.*pNaBpH - pCaBpMg.*pHBpCa.*pKBpH.*pMgBpNa.*pNaBpK + pCaBpMg.*pHBpCa.*pKBpNa.*pMgBpH.*pNaBpK - pCaBpMg.*pHBpCa.*pKBpNa.*pMgBpK.*pNaBpH - pCaBpMg.*pHBpNa.*pKBpH.*pMgBpK.*pNaBpCa + pCaBpMg.*pHBpNa.*pKBpH.*pMgBpCa.*pNaBpK - pCaBpMg.*pHBpNa.*pKBpCa.*pMgBpH.*pNaBpK + pCaBpMg.*pHBpNa.*pKBpCa.*pMgBpK.*pNaBpH - DCa.*DH.*DK.*DMg.*DNa);
dKdt = -(DH.*DMg.*PhiNa.*pCaBpNa.*pKBpCa + DCa.*DH.*PhiNa.*pKBpMg.*pMgBpNa + DCa.*DMg.*PhiNa.*pHBpNa.*pKBpH + DH.*DNa.*PhiMg.*pCaBpMg.*pKBpCa + DCa.*DH.*PhiMg.*pKBpNa.*pNaBpMg + DCa.*DNa.*PhiMg.*pHBpMg.*pKBpH - DH.*DMg.*PhiK.*pCaBpNa.*pNaBpCa - DH.*DNa.*PhiK.*pCaBpMg.*pMgBpCa - DCa.*DH.*PhiK.*pMgBpNa.*pNaBpMg - DMg.*DNa.*PhiK.*pCaBpH.*pHBpCa - DCa.*DMg.*PhiK.*pHBpNa.*pNaBpH - DCa.*DNa.*PhiK.*pHBpMg.*pMgBpH + DMg.*DNa.*PhiH.*pCaBpH.*pKBpCa + DCa.*DMg.*PhiH.*pKBpNa.*pNaBpH + DCa.*DNa.*PhiH.*pKBpMg.*pMgBpH + DH.*DMg.*PhiCa.*pKBpNa.*pNaBpCa + DH.*DNa.*PhiCa.*pKBpMg.*pMgBpCa + DMg.*DNa.*PhiCa.*pHBpCa.*pKBpH - DH.*PhiNa.*pCaBpNa.*pKBpMg.*pMgBpCa - DH.*PhiNa.*pCaBpMg.*pKBpCa.*pMgBpNa + DH.*PhiNa.*pCaBpMg.*pKBpNa.*pMgBpCa + DMg.*PhiNa.*pCaBpH.*pHBpCa.*pKBpNa - DMg.*PhiNa.*pCaBpH.*pHBpNa.*pKBpCa - DMg.*PhiNa.*pCaBpNa.*pHBpCa.*pKBpH - DCa.*PhiNa.*pHBpNa.*pKBpMg.*pMgBpH - DCa.*PhiNa.*pHBpMg.*pKBpH.*pMgBpNa + DCa.*PhiNa.*pHBpMg.*pKBpNa.*pMgBpH - DH.*PhiMg.*pCaBpNa.*pKBpCa.*pNaBpMg + DH.*PhiMg.*pCaBpNa.*pKBpMg.*pNaBpCa - DH.*PhiMg.*pCaBpMg.*pKBpNa.*pNaBpCa + DNa.*PhiMg.*pCaBpH.*pHBpCa.*pKBpMg - DNa.*PhiMg.*pCaBpH.*pHBpMg.*pKBpCa - DNa.*PhiMg.*pCaBpMg.*pHBpCa.*pKBpH - DCa.*PhiMg.*pHBpNa.*pKBpH.*pNaBpMg + DCa.*PhiMg.*pHBpNa.*pKBpMg.*pNaBpH - DCa.*PhiMg.*pHBpMg.*pKBpNa.*pNaBpH + DH.*PhiK.*pCaBpNa.*pMgBpCa.*pNaBpMg + DH.*PhiK.*pCaBpMg.*pMgBpNa.*pNaBpCa + DMg.*PhiK.*pCaBpH.*pHBpNa.*pNaBpCa + DMg.*PhiK.*pCaBpNa.*pHBpCa.*pNaBpH + DNa.*PhiK.*pCaBpH.*pHBpMg.*pMgBpCa + DNa.*PhiK.*pCaBpMg.*pHBpCa.*pMgBpH + DCa.*PhiK.*pHBpNa.*pMgBpH.*pNaBpMg + DCa.*PhiK.*pHBpMg.*pMgBpNa.*pNaBpH - DMg.*PhiH.*pCaBpH.*pKBpNa.*pNaBpCa + DMg.*PhiH.*pCaBpNa.*pKBpH.*pNaBpCa - DMg.*PhiH.*pCaBpNa.*pKBpCa.*pNaBpH - DNa.*PhiH.*pCaBpH.*pKBpMg.*pMgBpCa + DNa.*PhiH.*pCaBpMg.*pKBpH.*pMgBpCa - DNa.*PhiH.*pCaBpMg.*pKBpCa.*pMgBpH + DCa.*PhiH.*pKBpH.*pMgBpNa.*pNaBpMg - DCa.*PhiH.*pKBpNa.*pMgBpH.*pNaBpMg - DCa.*PhiH.*pKBpMg.*pMgBpNa.*pNaBpH + DH.*PhiCa.*pKBpCa.*pMgBpNa.*pNaBpMg - DH.*PhiCa.*pKBpNa.*pMgBpCa.*pNaBpMg - DH.*PhiCa.*pKBpMg.*pMgBpNa.*pNaBpCa - DMg.*PhiCa.*pHBpCa.*pKBpNa.*pNaBpH - DMg.*PhiCa.*pHBpNa.*pKBpH.*pNaBpCa + DMg.*PhiCa.*pHBpNa.*pKBpCa.*pNaBpH - DNa.*PhiCa.*pHBpCa.*pKBpMg.*pMgBpH - DNa.*PhiCa.*pHBpMg.*pKBpH.*pMgBpCa + DNa.*PhiCa.*pHBpMg.*pKBpCa.*pMgBpH - PhiNa.*pCaBpH.*pHBpCa.*pKBpMg.*pMgBpNa + PhiNa.*pCaBpH.*pHBpNa.*pKBpMg.*pMgBpCa + PhiNa.*pCaBpH.*pHBpMg.*pKBpCa.*pMgBpNa - PhiNa.*pCaBpH.*pHBpMg.*pKBpNa.*pMgBpCa + PhiNa.*pCaBpNa.*pHBpCa.*pKBpMg.*pMgBpH + PhiNa.*pCaBpNa.*pHBpMg.*pKBpH.*pMgBpCa - PhiNa.*pCaBpNa.*pHBpMg.*pKBpCa.*pMgBpH + PhiNa.*pCaBpMg.*pHBpCa.*pKBpH.*pMgBpNa - PhiNa.*pCaBpMg.*pHBpCa.*pKBpNa.*pMgBpH - PhiNa.*pCaBpMg.*pHBpNa.*pKBpH.*pMgBpCa + PhiNa.*pCaBpMg.*pHBpNa.*pKBpCa.*pMgBpH - PhiMg.*pCaBpH.*pHBpCa.*pKBpNa.*pNaBpMg + PhiMg.*pCaBpH.*pHBpNa.*pKBpCa.*pNaBpMg - PhiMg.*pCaBpH.*pHBpNa.*pKBpMg.*pNaBpCa + PhiMg.*pCaBpH.*pHBpMg.*pKBpNa.*pNaBpCa + PhiMg.*pCaBpNa.*pHBpCa.*pKBpH.*pNaBpMg - PhiMg.*pCaBpNa.*pHBpCa.*pKBpMg.*pNaBpH - PhiMg.*pCaBpNa.*pHBpMg.*pKBpH.*pNaBpCa + PhiMg.*pCaBpNa.*pHBpMg.*pKBpCa.*pNaBpH + PhiMg.*pCaBpMg.*pHBpCa.*pKBpNa.*pNaBpH + PhiMg.*pCaBpMg.*pHBpNa.*pKBpH.*pNaBpCa - PhiMg.*pCaBpMg.*pHBpNa.*pKBpCa.*pNaBpH + PhiK.*pCaBpH.*pHBpCa.*pMgBpNa.*pNaBpMg - PhiK.*pCaBpH.*pHBpNa.*pMgBpCa.*pNaBpMg - PhiK.*pCaBpH.*pHBpMg.*pMgBpNa.*pNaBpCa - PhiK.*pCaBpNa.*pHBpCa.*pMgBpH.*pNaBpMg + PhiK.*pCaBpNa.*pHBpMg.*pMgBpH.*pNaBpCa - PhiK.*pCaBpNa.*pHBpMg.*pMgBpCa.*pNaBpH - PhiK.*pCaBpMg.*pHBpCa.*pMgBpNa.*pNaBpH - PhiK.*pCaBpMg.*pHBpNa.*pMgBpH.*pNaBpCa + PhiK.*pCaBpMg.*pHBpNa.*pMgBpCa.*pNaBpH - PhiH.*pCaBpH.*pKBpCa.*pMgBpNa.*pNaBpMg + PhiH.*pCaBpH.*pKBpNa.*pMgBpCa.*pNaBpMg + PhiH.*pCaBpH.*pKBpMg.*pMgBpNa.*pNaBpCa - PhiH.*pCaBpNa.*pKBpH.*pMgBpCa.*pNaBpMg + PhiH.*pCaBpNa.*pKBpCa.*pMgBpH.*pNaBpMg - PhiH.*pCaBpNa.*pKBpMg.*pMgBpH.*pNaBpCa + PhiH.*pCaBpNa.*pKBpMg.*pMgBpCa.*pNaBpH - PhiH.*pCaBpMg.*pKBpH.*pMgBpNa.*pNaBpCa + PhiH.*pCaBpMg.*pKBpCa.*pMgBpNa.*pNaBpH + PhiH.*pCaBpMg.*pKBpNa.*pMgBpH.*pNaBpCa - PhiH.*pCaBpMg.*pKBpNa.*pMgBpCa.*pNaBpH - PhiCa.*pHBpCa.*pKBpH.*pMgBpNa.*pNaBpMg + PhiCa.*pHBpCa.*pKBpNa.*pMgBpH.*pNaBpMg + PhiCa.*pHBpCa.*pKBpMg.*pMgBpNa.*pNaBpH + PhiCa.*pHBpNa.*pKBpH.*pMgBpCa.*pNaBpMg - PhiCa.*pHBpNa.*pKBpCa.*pMgBpH.*pNaBpMg + PhiCa.*pHBpNa.*pKBpMg.*pMgBpH.*pNaBpCa - PhiCa.*pHBpNa.*pKBpMg.*pMgBpCa.*pNaBpH + PhiCa.*pHBpMg.*pKBpH.*pMgBpNa.*pNaBpCa - PhiCa.*pHBpMg.*pKBpCa.*pMgBpNa.*pNaBpH - PhiCa.*pHBpMg.*pKBpNa.*pMgBpH.*pNaBpCa + PhiCa.*pHBpMg.*pKBpNa.*pMgBpCa.*pNaBpH + DCa.*DH.*DMg.*DNa.*PhiK - DCa.*DH.*DMg.*PhiNa.*pKBpNa - DCa.*DH.*DNa.*PhiMg.*pKBpMg - DCa.*DMg.*DNa.*PhiH.*pKBpH - DH.*DMg.*DNa.*PhiCa.*pKBpCa)./(DH.*DK.*DMg.*pCaBpNa.*pNaBpCa + DH.*DK.*DNa.*pCaBpMg.*pMgBpCa + DCa.*DH.*DK.*pMgBpNa.*pNaBpMg + DK.*DMg.*DNa.*pCaBpH.*pHBpCa + DH.*DMg.*DNa.*pCaBpK.*pKBpCa + DCa.*DK.*DMg.*pHBpNa.*pNaBpH + DCa.*DH.*DMg.*pKBpNa.*pNaBpK + DCa.*DK.*DNa.*pHBpMg.*pMgBpH + DCa.*DH.*DNa.*pKBpMg.*pMgBpK + DCa.*DMg.*DNa.*pHBpK.*pKBpH - DH.*DK.*pCaBpNa.*pMgBpCa.*pNaBpMg - DH.*DK.*pCaBpMg.*pMgBpNa.*pNaBpCa - DK.*DMg.*pCaBpH.*pHBpNa.*pNaBpCa - DK.*DMg.*pCaBpNa.*pHBpCa.*pNaBpH - DH.*DMg.*pCaBpK.*pKBpNa.*pNaBpCa - DH.*DMg.*pCaBpNa.*pKBpCa.*pNaBpK - DK.*DNa.*pCaBpH.*pHBpMg.*pMgBpCa - DK.*DNa.*pCaBpMg.*pHBpCa.*pMgBpH - DH.*DNa.*pCaBpK.*pKBpMg.*pMgBpCa - DH.*DNa.*pCaBpMg.*pKBpCa.*pMgBpK - DCa.*DK.*pHBpNa.*pMgBpH.*pNaBpMg - DCa.*DK.*pHBpMg.*pMgBpNa.*pNaBpH - DCa.*DH.*pKBpNa.*pMgBpK.*pNaBpMg - DCa.*DH.*pKBpMg.*pMgBpNa.*pNaBpK - DMg.*DNa.*pCaBpH.*pHBpK.*pKBpCa - DMg.*DNa.*pCaBpK.*pHBpCa.*pKBpH - DCa.*DMg.*pHBpK.*pKBpNa.*pNaBpH - DCa.*DMg.*pHBpNa.*pKBpH.*pNaBpK - DCa.*DNa.*pHBpK.*pKBpMg.*pMgBpH - DCa.*DNa.*pHBpMg.*pKBpH.*pMgBpK - DK.*pCaBpH.*pHBpCa.*pMgBpNa.*pNaBpMg + DK.*pCaBpH.*pHBpNa.*pMgBpCa.*pNaBpMg + DK.*pCaBpH.*pHBpMg.*pMgBpNa.*pNaBpCa + DK.*pCaBpNa.*pHBpCa.*pMgBpH.*pNaBpMg - DK.*pCaBpNa.*pHBpMg.*pMgBpH.*pNaBpCa + DK.*pCaBpNa.*pHBpMg.*pMgBpCa.*pNaBpH + DK.*pCaBpMg.*pHBpCa.*pMgBpNa.*pNaBpH + DK.*pCaBpMg.*pHBpNa.*pMgBpH.*pNaBpCa - DK.*pCaBpMg.*pHBpNa.*pMgBpCa.*pNaBpH - DH.*pCaBpK.*pKBpCa.*pMgBpNa.*pNaBpMg + DH.*pCaBpK.*pKBpNa.*pMgBpCa.*pNaBpMg + DH.*pCaBpK.*pKBpMg.*pMgBpNa.*pNaBpCa + DH.*pCaBpNa.*pKBpCa.*pMgBpK.*pNaBpMg - DH.*pCaBpNa.*pKBpMg.*pMgBpK.*pNaBpCa + DH.*pCaBpNa.*pKBpMg.*pMgBpCa.*pNaBpK + DH.*pCaBpMg.*pKBpCa.*pMgBpNa.*pNaBpK + DH.*pCaBpMg.*pKBpNa.*pMgBpK.*pNaBpCa - DH.*pCaBpMg.*pKBpNa.*pMgBpCa.*pNaBpK + DMg.*pCaBpH.*pHBpK.*pKBpNa.*pNaBpCa - DMg.*pCaBpH.*pHBpCa.*pKBpNa.*pNaBpK + DMg.*pCaBpH.*pHBpNa.*pKBpCa.*pNaBpK + DMg.*pCaBpK.*pHBpCa.*pKBpNa.*pNaBpH + DMg.*pCaBpK.*pHBpNa.*pKBpH.*pNaBpCa - DMg.*pCaBpK.*pHBpNa.*pKBpCa.*pNaBpH - DMg.*pCaBpNa.*pHBpK.*pKBpH.*pNaBpCa + DMg.*pCaBpNa.*pHBpK.*pKBpCa.*pNaBpH + DMg.*pCaBpNa.*pHBpCa.*pKBpH.*pNaBpK + DNa.*pCaBpH.*pHBpK.*pKBpMg.*pMgBpCa - DNa.*pCaBpH.*pHBpCa.*pKBpMg.*pMgBpK + DNa.*pCaBpH.*pHBpMg.*pKBpCa.*pMgBpK + DNa.*pCaBpK.*pHBpCa.*pKBpMg.*pMgBpH + DNa.*pCaBpK.*pHBpMg.*pKBpH.*pMgBpCa - DNa.*pCaBpK.*pHBpMg.*pKBpCa.*pMgBpH - DNa.*pCaBpMg.*pHBpK.*pKBpH.*pMgBpCa + DNa.*pCaBpMg.*pHBpK.*pKBpCa.*pMgBpH + DNa.*pCaBpMg.*pHBpCa.*pKBpH.*pMgBpK - DCa.*pHBpK.*pKBpH.*pMgBpNa.*pNaBpMg + DCa.*pHBpK.*pKBpNa.*pMgBpH.*pNaBpMg + DCa.*pHBpK.*pKBpMg.*pMgBpNa.*pNaBpH + DCa.*pHBpNa.*pKBpH.*pMgBpK.*pNaBpMg + DCa.*pHBpNa.*pKBpMg.*pMgBpH.*pNaBpK - DCa.*pHBpNa.*pKBpMg.*pMgBpK.*pNaBpH + DCa.*pHBpMg.*pKBpH.*pMgBpNa.*pNaBpK - DCa.*pHBpMg.*pKBpNa.*pMgBpH.*pNaBpK + DCa.*pHBpMg.*pKBpNa.*pMgBpK.*pNaBpH + pCaBpH.*pHBpK.*pKBpCa.*pMgBpNa.*pNaBpMg - pCaBpH.*pHBpK.*pKBpNa.*pMgBpCa.*pNaBpMg - pCaBpH.*pHBpK.*pKBpMg.*pMgBpNa.*pNaBpCa + pCaBpH.*pHBpCa.*pKBpNa.*pMgBpK.*pNaBpMg + pCaBpH.*pHBpCa.*pKBpMg.*pMgBpNa.*pNaBpK - pCaBpH.*pHBpNa.*pKBpCa.*pMgBpK.*pNaBpMg + pCaBpH.*pHBpNa.*pKBpMg.*pMgBpK.*pNaBpCa - pCaBpH.*pHBpNa.*pKBpMg.*pMgBpCa.*pNaBpK - pCaBpH.*pHBpMg.*pKBpCa.*pMgBpNa.*pNaBpK - pCaBpH.*pHBpMg.*pKBpNa.*pMgBpK.*pNaBpCa + pCaBpH.*pHBpMg.*pKBpNa.*pMgBpCa.*pNaBpK + pCaBpK.*pHBpCa.*pKBpH.*pMgBpNa.*pNaBpMg - pCaBpK.*pHBpCa.*pKBpNa.*pMgBpH.*pNaBpMg - pCaBpK.*pHBpCa.*pKBpMg.*pMgBpNa.*pNaBpH - pCaBpK.*pHBpNa.*pKBpH.*pMgBpCa.*pNaBpMg + pCaBpK.*pHBpNa.*pKBpCa.*pMgBpH.*pNaBpMg - pCaBpK.*pHBpNa.*pKBpMg.*pMgBpH.*pNaBpCa + pCaBpK.*pHBpNa.*pKBpMg.*pMgBpCa.*pNaBpH - pCaBpK.*pHBpMg.*pKBpH.*pMgBpNa.*pNaBpCa + pCaBpK.*pHBpMg.*pKBpCa.*pMgBpNa.*pNaBpH + pCaBpK.*pHBpMg.*pKBpNa.*pMgBpH.*pNaBpCa - pCaBpK.*pHBpMg.*pKBpNa.*pMgBpCa.*pNaBpH + pCaBpNa.*pHBpK.*pKBpH.*pMgBpCa.*pNaBpMg - pCaBpNa.*pHBpK.*pKBpCa.*pMgBpH.*pNaBpMg + pCaBpNa.*pHBpK.*pKBpMg.*pMgBpH.*pNaBpCa - pCaBpNa.*pHBpK.*pKBpMg.*pMgBpCa.*pNaBpH - pCaBpNa.*pHBpCa.*pKBpH.*pMgBpK.*pNaBpMg - pCaBpNa.*pHBpCa.*pKBpMg.*pMgBpH.*pNaBpK + pCaBpNa.*pHBpCa.*pKBpMg.*pMgBpK.*pNaBpH + pCaBpNa.*pHBpMg.*pKBpH.*pMgBpK.*pNaBpCa - pCaBpNa.*pHBpMg.*pKBpH.*pMgBpCa.*pNaBpK + pCaBpNa.*pHBpMg.*pKBpCa.*pMgBpH.*pNaBpK - pCaBpNa.*pHBpMg.*pKBpCa.*pMgBpK.*pNaBpH + pCaBpMg.*pHBpK.*pKBpH.*pMgBpNa.*pNaBpCa - pCaBpMg.*pHBpK.*pKBpCa.*pMgBpNa.*pNaBpH - pCaBpMg.*pHBpK.*pKBpNa.*pMgBpH.*pNaBpCa + pCaBpMg.*pHBpK.*pKBpNa.*pMgBpCa.*pNaBpH - pCaBpMg.*pHBpCa.*pKBpH.*pMgBpNa.*pNaBpK + pCaBpMg.*pHBpCa.*pKBpNa.*pMgBpH.*pNaBpK - pCaBpMg.*pHBpCa.*pKBpNa.*pMgBpK.*pNaBpH - pCaBpMg.*pHBpNa.*pKBpH.*pMgBpK.*pNaBpCa + pCaBpMg.*pHBpNa.*pKBpH.*pMgBpCa.*pNaBpK - pCaBpMg.*pHBpNa.*pKBpCa.*pMgBpH.*pNaBpK + pCaBpMg.*pHBpNa.*pKBpCa.*pMgBpK.*pNaBpH - DCa.*DH.*DK.*DMg.*DNa);
dNadt = (DH.*DK.*PhiNa.*pCaBpMg.*pMgBpCa + DK.*DMg.*PhiNa.*pCaBpH.*pHBpCa + DH.*DMg.*PhiNa.*pCaBpK.*pKBpCa + DCa.*DK.*PhiNa.*pHBpMg.*pMgBpH + DCa.*DH.*PhiNa.*pKBpMg.*pMgBpK + DCa.*DMg.*PhiNa.*pHBpK.*pKBpH - DH.*DK.*PhiMg.*pCaBpMg.*pNaBpCa - DCa.*DK.*PhiMg.*pHBpMg.*pNaBpH - DCa.*DH.*PhiMg.*pKBpMg.*pNaBpK - DH.*DMg.*PhiK.*pCaBpK.*pNaBpCa - DCa.*DH.*PhiK.*pMgBpK.*pNaBpMg - DCa.*DMg.*PhiK.*pHBpK.*pNaBpH - DK.*DMg.*PhiH.*pCaBpH.*pNaBpCa - DCa.*DK.*PhiH.*pMgBpH.*pNaBpMg - DCa.*DMg.*PhiH.*pKBpH.*pNaBpK - DH.*DK.*PhiCa.*pMgBpCa.*pNaBpMg - DK.*DMg.*PhiCa.*pHBpCa.*pNaBpH - DH.*DMg.*PhiCa.*pKBpCa.*pNaBpK - DK.*PhiNa.*pCaBpH.*pHBpMg.*pMgBpCa - DK.*PhiNa.*pCaBpMg.*pHBpCa.*pMgBpH - DH.*PhiNa.*pCaBpK.*pKBpMg.*pMgBpCa - DH.*PhiNa.*pCaBpMg.*pKBpCa.*pMgBpK - DMg.*PhiNa.*pCaBpH.*pHBpK.*pKBpCa - DMg.*PhiNa.*pCaBpK.*pHBpCa.*pKBpH - DCa.*PhiNa.*pHBpK.*pKBpMg.*pMgBpH - DCa.*PhiNa.*pHBpMg.*pKBpH.*pMgBpK - DK.*PhiMg.*pCaBpH.*pHBpCa.*pNaBpMg + DK.*PhiMg.*pCaBpH.*pHBpMg.*pNaBpCa + DK.*PhiMg.*pCaBpMg.*pHBpCa.*pNaBpH - DH.*PhiMg.*pCaBpK.*pKBpCa.*pNaBpMg + DH.*PhiMg.*pCaBpK.*pKBpMg.*pNaBpCa + DH.*PhiMg.*pCaBpMg.*pKBpCa.*pNaBpK - DCa.*PhiMg.*pHBpK.*pKBpH.*pNaBpMg + DCa.*PhiMg.*pHBpK.*pKBpMg.*pNaBpH + DCa.*PhiMg.*pHBpMg.*pKBpH.*pNaBpK + DH.*PhiK.*pCaBpK.*pMgBpCa.*pNaBpMg + DH.*PhiK.*pCaBpMg.*pMgBpK.*pNaBpCa - DH.*PhiK.*pCaBpMg.*pMgBpCa.*pNaBpK + DMg.*PhiK.*pCaBpH.*pHBpK.*pNaBpCa - DMg.*PhiK.*pCaBpH.*pHBpCa.*pNaBpK + DMg.*PhiK.*pCaBpK.*pHBpCa.*pNaBpH + DCa.*PhiK.*pHBpK.*pMgBpH.*pNaBpMg - DCa.*PhiK.*pHBpMg.*pMgBpH.*pNaBpK + DCa.*PhiK.*pHBpMg.*pMgBpK.*pNaBpH + DK.*PhiH.*pCaBpH.*pMgBpCa.*pNaBpMg + DK.*PhiH.*pCaBpMg.*pMgBpH.*pNaBpCa - DK.*PhiH.*pCaBpMg.*pMgBpCa.*pNaBpH + DMg.*PhiH.*pCaBpH.*pKBpCa.*pNaBpK + DMg.*PhiH.*pCaBpK.*pKBpH.*pNaBpCa - DMg.*PhiH.*pCaBpK.*pKBpCa.*pNaBpH + DCa.*PhiH.*pKBpH.*pMgBpK.*pNaBpMg + DCa.*PhiH.*pKBpMg.*pMgBpH.*pNaBpK - DCa.*PhiH.*pKBpMg.*pMgBpK.*pNaBpH + DK.*PhiCa.*pHBpCa.*pMgBpH.*pNaBpMg - DK.*PhiCa.*pHBpMg.*pMgBpH.*pNaBpCa + DK.*PhiCa.*pHBpMg.*pMgBpCa.*pNaBpH + DH.*PhiCa.*pKBpCa.*pMgBpK.*pNaBpMg - DH.*PhiCa.*pKBpMg.*pMgBpK.*pNaBpCa + DH.*PhiCa.*pKBpMg.*pMgBpCa.*pNaBpK - DMg.*PhiCa.*pHBpK.*pKBpH.*pNaBpCa + DMg.*PhiCa.*pHBpK.*pKBpCa.*pNaBpH + DMg.*PhiCa.*pHBpCa.*pKBpH.*pNaBpK + PhiNa.*pCaBpH.*pHBpK.*pKBpMg.*pMgBpCa - PhiNa.*pCaBpH.*pHBpCa.*pKBpMg.*pMgBpK + PhiNa.*pCaBpH.*pHBpMg.*pKBpCa.*pMgBpK + PhiNa.*pCaBpK.*pHBpCa.*pKBpMg.*pMgBpH + PhiNa.*pCaBpK.*pHBpMg.*pKBpH.*pMgBpCa - PhiNa.*pCaBpK.*pHBpMg.*pKBpCa.*pMgBpH - PhiNa.*pCaBpMg.*pHBpK.*pKBpH.*pMgBpCa + PhiNa.*pCaBpMg.*pHBpK.*pKBpCa.*pMgBpH + PhiNa.*pCaBpMg.*pHBpCa.*pKBpH.*pMgBpK + PhiMg.*pCaBpH.*pHBpK.*pKBpCa.*pNaBpMg - PhiMg.*pCaBpH.*pHBpK.*pKBpMg.*pNaBpCa + PhiMg.*pCaBpH.*pHBpCa.*pKBpMg.*pNaBpK - PhiMg.*pCaBpH.*pHBpMg.*pKBpCa.*pNaBpK + PhiMg.*pCaBpK.*pHBpCa.*pKBpH.*pNaBpMg - PhiMg.*pCaBpK.*pHBpCa.*pKBpMg.*pNaBpH - PhiMg.*pCaBpK.*pHBpMg.*pKBpH.*pNaBpCa + PhiMg.*pCaBpK.*pHBpMg.*pKBpCa.*pNaBpH + PhiMg.*pCaBpMg.*pHBpK.*pKBpH.*pNaBpCa - PhiMg.*pCaBpMg.*pHBpK.*pKBpCa.*pNaBpH - PhiMg.*pCaBpMg.*pHBpCa.*pKBpH.*pNaBpK - PhiK.*pCaBpH.*pHBpK.*pMgBpCa.*pNaBpMg + PhiK.*pCaBpH.*pHBpCa.*pMgBpK.*pNaBpMg - PhiK.*pCaBpH.*pHBpMg.*pMgBpK.*pNaBpCa + PhiK.*pCaBpH.*pHBpMg.*pMgBpCa.*pNaBpK - PhiK.*pCaBpK.*pHBpCa.*pMgBpH.*pNaBpMg + PhiK.*pCaBpK.*pHBpMg.*pMgBpH.*pNaBpCa - PhiK.*pCaBpK.*pHBpMg.*pMgBpCa.*pNaBpH - PhiK.*pCaBpMg.*pHBpK.*pMgBpH.*pNaBpCa + PhiK.*pCaBpMg.*pHBpK.*pMgBpCa.*pNaBpH + PhiK.*pCaBpMg.*pHBpCa.*pMgBpH.*pNaBpK - PhiK.*pCaBpMg.*pHBpCa.*pMgBpK.*pNaBpH - PhiH.*pCaBpH.*pKBpCa.*pMgBpK.*pNaBpMg + PhiH.*pCaBpH.*pKBpMg.*pMgBpK.*pNaBpCa - PhiH.*pCaBpH.*pKBpMg.*pMgBpCa.*pNaBpK - PhiH.*pCaBpK.*pKBpH.*pMgBpCa.*pNaBpMg + PhiH.*pCaBpK.*pKBpCa.*pMgBpH.*pNaBpMg - PhiH.*pCaBpK.*pKBpMg.*pMgBpH.*pNaBpCa + PhiH.*pCaBpK.*pKBpMg.*pMgBpCa.*pNaBpH - PhiH.*pCaBpMg.*pKBpH.*pMgBpK.*pNaBpCa + PhiH.*pCaBpMg.*pKBpH.*pMgBpCa.*pNaBpK - PhiH.*pCaBpMg.*pKBpCa.*pMgBpH.*pNaBpK + PhiH.*pCaBpMg.*pKBpCa.*pMgBpK.*pNaBpH + PhiCa.*pHBpK.*pKBpH.*pMgBpCa.*pNaBpMg - PhiCa.*pHBpK.*pKBpCa.*pMgBpH.*pNaBpMg + PhiCa.*pHBpK.*pKBpMg.*pMgBpH.*pNaBpCa - PhiCa.*pHBpK.*pKBpMg.*pMgBpCa.*pNaBpH - PhiCa.*pHBpCa.*pKBpH.*pMgBpK.*pNaBpMg - PhiCa.*pHBpCa.*pKBpMg.*pMgBpH.*pNaBpK + PhiCa.*pHBpCa.*pKBpMg.*pMgBpK.*pNaBpH + PhiCa.*pHBpMg.*pKBpH.*pMgBpK.*pNaBpCa - PhiCa.*pHBpMg.*pKBpH.*pMgBpCa.*pNaBpK + PhiCa.*pHBpMg.*pKBpCa.*pMgBpH.*pNaBpK - PhiCa.*pHBpMg.*pKBpCa.*pMgBpK.*pNaBpH - DCa.*DH.*DK.*DMg.*PhiNa + DCa.*DH.*DK.*PhiMg.*pNaBpMg + DCa.*DH.*DMg.*PhiK.*pNaBpK + DCa.*DK.*DMg.*PhiH.*pNaBpH + DH.*DK.*DMg.*PhiCa.*pNaBpCa)./(DH.*DK.*DMg.*pCaBpNa.*pNaBpCa + DH.*DK.*DNa.*pCaBpMg.*pMgBpCa + DCa.*DH.*DK.*pMgBpNa.*pNaBpMg + DK.*DMg.*DNa.*pCaBpH.*pHBpCa + DH.*DMg.*DNa.*pCaBpK.*pKBpCa + DCa.*DK.*DMg.*pHBpNa.*pNaBpH + DCa.*DH.*DMg.*pKBpNa.*pNaBpK + DCa.*DK.*DNa.*pHBpMg.*pMgBpH + DCa.*DH.*DNa.*pKBpMg.*pMgBpK + DCa.*DMg.*DNa.*pHBpK.*pKBpH - DH.*DK.*pCaBpNa.*pMgBpCa.*pNaBpMg - DH.*DK.*pCaBpMg.*pMgBpNa.*pNaBpCa - DK.*DMg.*pCaBpH.*pHBpNa.*pNaBpCa - DK.*DMg.*pCaBpNa.*pHBpCa.*pNaBpH - DH.*DMg.*pCaBpK.*pKBpNa.*pNaBpCa - DH.*DMg.*pCaBpNa.*pKBpCa.*pNaBpK - DK.*DNa.*pCaBpH.*pHBpMg.*pMgBpCa - DK.*DNa.*pCaBpMg.*pHBpCa.*pMgBpH - DH.*DNa.*pCaBpK.*pKBpMg.*pMgBpCa - DH.*DNa.*pCaBpMg.*pKBpCa.*pMgBpK - DCa.*DK.*pHBpNa.*pMgBpH.*pNaBpMg - DCa.*DK.*pHBpMg.*pMgBpNa.*pNaBpH - DCa.*DH.*pKBpNa.*pMgBpK.*pNaBpMg - DCa.*DH.*pKBpMg.*pMgBpNa.*pNaBpK - DMg.*DNa.*pCaBpH.*pHBpK.*pKBpCa - DMg.*DNa.*pCaBpK.*pHBpCa.*pKBpH - DCa.*DMg.*pHBpK.*pKBpNa.*pNaBpH - DCa.*DMg.*pHBpNa.*pKBpH.*pNaBpK - DCa.*DNa.*pHBpK.*pKBpMg.*pMgBpH - DCa.*DNa.*pHBpMg.*pKBpH.*pMgBpK - DK.*pCaBpH.*pHBpCa.*pMgBpNa.*pNaBpMg + DK.*pCaBpH.*pHBpNa.*pMgBpCa.*pNaBpMg + DK.*pCaBpH.*pHBpMg.*pMgBpNa.*pNaBpCa + DK.*pCaBpNa.*pHBpCa.*pMgBpH.*pNaBpMg - DK.*pCaBpNa.*pHBpMg.*pMgBpH.*pNaBpCa + DK.*pCaBpNa.*pHBpMg.*pMgBpCa.*pNaBpH + DK.*pCaBpMg.*pHBpCa.*pMgBpNa.*pNaBpH + DK.*pCaBpMg.*pHBpNa.*pMgBpH.*pNaBpCa - DK.*pCaBpMg.*pHBpNa.*pMgBpCa.*pNaBpH - DH.*pCaBpK.*pKBpCa.*pMgBpNa.*pNaBpMg + DH.*pCaBpK.*pKBpNa.*pMgBpCa.*pNaBpMg + DH.*pCaBpK.*pKBpMg.*pMgBpNa.*pNaBpCa + DH.*pCaBpNa.*pKBpCa.*pMgBpK.*pNaBpMg - DH.*pCaBpNa.*pKBpMg.*pMgBpK.*pNaBpCa + DH.*pCaBpNa.*pKBpMg.*pMgBpCa.*pNaBpK + DH.*pCaBpMg.*pKBpCa.*pMgBpNa.*pNaBpK + DH.*pCaBpMg.*pKBpNa.*pMgBpK.*pNaBpCa - DH.*pCaBpMg.*pKBpNa.*pMgBpCa.*pNaBpK + DMg.*pCaBpH.*pHBpK.*pKBpNa.*pNaBpCa - DMg.*pCaBpH.*pHBpCa.*pKBpNa.*pNaBpK + DMg.*pCaBpH.*pHBpNa.*pKBpCa.*pNaBpK + DMg.*pCaBpK.*pHBpCa.*pKBpNa.*pNaBpH + DMg.*pCaBpK.*pHBpNa.*pKBpH.*pNaBpCa - DMg.*pCaBpK.*pHBpNa.*pKBpCa.*pNaBpH - DMg.*pCaBpNa.*pHBpK.*pKBpH.*pNaBpCa + DMg.*pCaBpNa.*pHBpK.*pKBpCa.*pNaBpH + DMg.*pCaBpNa.*pHBpCa.*pKBpH.*pNaBpK + DNa.*pCaBpH.*pHBpK.*pKBpMg.*pMgBpCa - DNa.*pCaBpH.*pHBpCa.*pKBpMg.*pMgBpK + DNa.*pCaBpH.*pHBpMg.*pKBpCa.*pMgBpK + DNa.*pCaBpK.*pHBpCa.*pKBpMg.*pMgBpH + DNa.*pCaBpK.*pHBpMg.*pKBpH.*pMgBpCa - DNa.*pCaBpK.*pHBpMg.*pKBpCa.*pMgBpH - DNa.*pCaBpMg.*pHBpK.*pKBpH.*pMgBpCa + DNa.*pCaBpMg.*pHBpK.*pKBpCa.*pMgBpH + DNa.*pCaBpMg.*pHBpCa.*pKBpH.*pMgBpK - DCa.*pHBpK.*pKBpH.*pMgBpNa.*pNaBpMg + DCa.*pHBpK.*pKBpNa.*pMgBpH.*pNaBpMg + DCa.*pHBpK.*pKBpMg.*pMgBpNa.*pNaBpH + DCa.*pHBpNa.*pKBpH.*pMgBpK.*pNaBpMg + DCa.*pHBpNa.*pKBpMg.*pMgBpH.*pNaBpK - DCa.*pHBpNa.*pKBpMg.*pMgBpK.*pNaBpH + DCa.*pHBpMg.*pKBpH.*pMgBpNa.*pNaBpK - DCa.*pHBpMg.*pKBpNa.*pMgBpH.*pNaBpK + DCa.*pHBpMg.*pKBpNa.*pMgBpK.*pNaBpH + pCaBpH.*pHBpK.*pKBpCa.*pMgBpNa.*pNaBpMg - pCaBpH.*pHBpK.*pKBpNa.*pMgBpCa.*pNaBpMg - pCaBpH.*pHBpK.*pKBpMg.*pMgBpNa.*pNaBpCa + pCaBpH.*pHBpCa.*pKBpNa.*pMgBpK.*pNaBpMg + pCaBpH.*pHBpCa.*pKBpMg.*pMgBpNa.*pNaBpK - pCaBpH.*pHBpNa.*pKBpCa.*pMgBpK.*pNaBpMg + pCaBpH.*pHBpNa.*pKBpMg.*pMgBpK.*pNaBpCa - pCaBpH.*pHBpNa.*pKBpMg.*pMgBpCa.*pNaBpK - pCaBpH.*pHBpMg.*pKBpCa.*pMgBpNa.*pNaBpK - pCaBpH.*pHBpMg.*pKBpNa.*pMgBpK.*pNaBpCa + pCaBpH.*pHBpMg.*pKBpNa.*pMgBpCa.*pNaBpK + pCaBpK.*pHBpCa.*pKBpH.*pMgBpNa.*pNaBpMg - pCaBpK.*pHBpCa.*pKBpNa.*pMgBpH.*pNaBpMg - pCaBpK.*pHBpCa.*pKBpMg.*pMgBpNa.*pNaBpH - pCaBpK.*pHBpNa.*pKBpH.*pMgBpCa.*pNaBpMg + pCaBpK.*pHBpNa.*pKBpCa.*pMgBpH.*pNaBpMg - pCaBpK.*pHBpNa.*pKBpMg.*pMgBpH.*pNaBpCa + pCaBpK.*pHBpNa.*pKBpMg.*pMgBpCa.*pNaBpH - pCaBpK.*pHBpMg.*pKBpH.*pMgBpNa.*pNaBpCa + pCaBpK.*pHBpMg.*pKBpCa.*pMgBpNa.*pNaBpH + pCaBpK.*pHBpMg.*pKBpNa.*pMgBpH.*pNaBpCa - pCaBpK.*pHBpMg.*pKBpNa.*pMgBpCa.*pNaBpH + pCaBpNa.*pHBpK.*pKBpH.*pMgBpCa.*pNaBpMg - pCaBpNa.*pHBpK.*pKBpCa.*pMgBpH.*pNaBpMg + pCaBpNa.*pHBpK.*pKBpMg.*pMgBpH.*pNaBpCa - pCaBpNa.*pHBpK.*pKBpMg.*pMgBpCa.*pNaBpH - pCaBpNa.*pHBpCa.*pKBpH.*pMgBpK.*pNaBpMg - pCaBpNa.*pHBpCa.*pKBpMg.*pMgBpH.*pNaBpK + pCaBpNa.*pHBpCa.*pKBpMg.*pMgBpK.*pNaBpH + pCaBpNa.*pHBpMg.*pKBpH.*pMgBpK.*pNaBpCa - pCaBpNa.*pHBpMg.*pKBpH.*pMgBpCa.*pNaBpK + pCaBpNa.*pHBpMg.*pKBpCa.*pMgBpH.*pNaBpK - pCaBpNa.*pHBpMg.*pKBpCa.*pMgBpK.*pNaBpH + pCaBpMg.*pHBpK.*pKBpH.*pMgBpNa.*pNaBpCa - pCaBpMg.*pHBpK.*pKBpCa.*pMgBpNa.*pNaBpH - pCaBpMg.*pHBpK.*pKBpNa.*pMgBpH.*pNaBpCa + pCaBpMg.*pHBpK.*pKBpNa.*pMgBpCa.*pNaBpH - pCaBpMg.*pHBpCa.*pKBpH.*pMgBpNa.*pNaBpK + pCaBpMg.*pHBpCa.*pKBpNa.*pMgBpH.*pNaBpK - pCaBpMg.*pHBpCa.*pKBpNa.*pMgBpK.*pNaBpH - pCaBpMg.*pHBpNa.*pKBpH.*pMgBpK.*pNaBpCa + pCaBpMg.*pHBpNa.*pKBpH.*pMgBpCa.*pNaBpK - pCaBpMg.*pHBpNa.*pKBpCa.*pMgBpH.*pNaBpK + pCaBpMg.*pHBpNa.*pKBpCa.*pMgBpK.*pNaBpH - DCa.*DH.*DK.*DMg.*DNa);
dMgdt = -(DH.*DK.*PhiNa.*pCaBpNa.*pMgBpCa + DCa.*DK.*PhiNa.*pHBpNa.*pMgBpH + DCa.*DH.*PhiNa.*pKBpNa.*pMgBpK - DH.*DK.*PhiMg.*pCaBpNa.*pNaBpCa - DK.*DNa.*PhiMg.*pCaBpH.*pHBpCa - DH.*DNa.*PhiMg.*pCaBpK.*pKBpCa - DCa.*DK.*PhiMg.*pHBpNa.*pNaBpH - DCa.*DH.*PhiMg.*pKBpNa.*pNaBpK - DCa.*DNa.*PhiMg.*pHBpK.*pKBpH + DH.*DNa.*PhiK.*pCaBpK.*pMgBpCa + DCa.*DH.*PhiK.*pMgBpNa.*pNaBpK + DCa.*DNa.*PhiK.*pHBpK.*pMgBpH + DK.*DNa.*PhiH.*pCaBpH.*pMgBpCa + DCa.*DK.*PhiH.*pMgBpNa.*pNaBpH + DCa.*DNa.*PhiH.*pKBpH.*pMgBpK + DH.*DK.*PhiCa.*pMgBpNa.*pNaBpCa + DK.*DNa.*PhiCa.*pHBpCa.*pMgBpH + DH.*DNa.*PhiCa.*pKBpCa.*pMgBpK + DK.*PhiNa.*pCaBpH.*pHBpCa.*pMgBpNa - DK.*PhiNa.*pCaBpH.*pHBpNa.*pMgBpCa - DK.*PhiNa.*pCaBpNa.*pHBpCa.*pMgBpH + DH.*PhiNa.*pCaBpK.*pKBpCa.*pMgBpNa - DH.*PhiNa.*pCaBpK.*pKBpNa.*pMgBpCa - DH.*PhiNa.*pCaBpNa.*pKBpCa.*pMgBpK + DCa.*PhiNa.*pHBpK.*pKBpH.*pMgBpNa - DCa.*PhiNa.*pHBpK.*pKBpNa.*pMgBpH - DCa.*PhiNa.*pHBpNa.*pKBpH.*pMgBpK + DK.*PhiMg.*pCaBpH.*pHBpNa.*pNaBpCa + DK.*PhiMg.*pCaBpNa.*pHBpCa.*pNaBpH + DH.*PhiMg.*pCaBpK.*pKBpNa.*pNaBpCa + DH.*PhiMg.*pCaBpNa.*pKBpCa.*pNaBpK + DNa.*PhiMg.*pCaBpH.*pHBpK.*pKBpCa + DNa.*PhiMg.*pCaBpK.*pHBpCa.*pKBpH + DCa.*PhiMg.*pHBpK.*pKBpNa.*pNaBpH + DCa.*PhiMg.*pHBpNa.*pKBpH.*pNaBpK - DH.*PhiK.*pCaBpK.*pMgBpNa.*pNaBpCa + DH.*PhiK.*pCaBpNa.*pMgBpK.*pNaBpCa - DH.*PhiK.*pCaBpNa.*pMgBpCa.*pNaBpK - DNa.*PhiK.*pCaBpH.*pHBpK.*pMgBpCa + DNa.*PhiK.*pCaBpH.*pHBpCa.*pMgBpK - DNa.*PhiK.*pCaBpK.*pHBpCa.*pMgBpH - DCa.*PhiK.*pHBpK.*pMgBpNa.*pNaBpH - DCa.*PhiK.*pHBpNa.*pMgBpH.*pNaBpK + DCa.*PhiK.*pHBpNa.*pMgBpK.*pNaBpH - DK.*PhiH.*pCaBpH.*pMgBpNa.*pNaBpCa + DK.*PhiH.*pCaBpNa.*pMgBpH.*pNaBpCa - DK.*PhiH.*pCaBpNa.*pMgBpCa.*pNaBpH - DNa.*PhiH.*pCaBpH.*pKBpCa.*pMgBpK - DNa.*PhiH.*pCaBpK.*pKBpH.*pMgBpCa + DNa.*PhiH.*pCaBpK.*pKBpCa.*pMgBpH - DCa.*PhiH.*pKBpH.*pMgBpNa.*pNaBpK + DCa.*PhiH.*pKBpNa.*pMgBpH.*pNaBpK - DCa.*PhiH.*pKBpNa.*pMgBpK.*pNaBpH - DK.*PhiCa.*pHBpCa.*pMgBpNa.*pNaBpH - DK.*PhiCa.*pHBpNa.*pMgBpH.*pNaBpCa + DK.*PhiCa.*pHBpNa.*pMgBpCa.*pNaBpH - DH.*PhiCa.*pKBpCa.*pMgBpNa.*pNaBpK - DH.*PhiCa.*pKBpNa.*pMgBpK.*pNaBpCa + DH.*PhiCa.*pKBpNa.*pMgBpCa.*pNaBpK + DNa.*PhiCa.*pHBpK.*pKBpH.*pMgBpCa - DNa.*PhiCa.*pHBpK.*pKBpCa.*pMgBpH - DNa.*PhiCa.*pHBpCa.*pKBpH.*pMgBpK - PhiNa.*pCaBpH.*pHBpK.*pKBpCa.*pMgBpNa + PhiNa.*pCaBpH.*pHBpK.*pKBpNa.*pMgBpCa - PhiNa.*pCaBpH.*pHBpCa.*pKBpNa.*pMgBpK + PhiNa.*pCaBpH.*pHBpNa.*pKBpCa.*pMgBpK - PhiNa.*pCaBpK.*pHBpCa.*pKBpH.*pMgBpNa + PhiNa.*pCaBpK.*pHBpCa.*pKBpNa.*pMgBpH + PhiNa.*pCaBpK.*pHBpNa.*pKBpH.*pMgBpCa - PhiNa.*pCaBpK.*pHBpNa.*pKBpCa.*pMgBpH - PhiNa.*pCaBpNa.*pHBpK.*pKBpH.*pMgBpCa + PhiNa.*pCaBpNa.*pHBpK.*pKBpCa.*pMgBpH + PhiNa.*pCaBpNa.*pHBpCa.*pKBpH.*pMgBpK - PhiMg.*pCaBpH.*pHBpK.*pKBpNa.*pNaBpCa + PhiMg.*pCaBpH.*pHBpCa.*pKBpNa.*pNaBpK - PhiMg.*pCaBpH.*pHBpNa.*pKBpCa.*pNaBpK - PhiMg.*pCaBpK.*pHBpCa.*pKBpNa.*pNaBpH - PhiMg.*pCaBpK.*pHBpNa.*pKBpH.*pNaBpCa + PhiMg.*pCaBpK.*pHBpNa.*pKBpCa.*pNaBpH + PhiMg.*pCaBpNa.*pHBpK.*pKBpH.*pNaBpCa - PhiMg.*pCaBpNa.*pHBpK.*pKBpCa.*pNaBpH - PhiMg.*pCaBpNa.*pHBpCa.*pKBpH.*pNaBpK + PhiK.*pCaBpH.*pHBpK.*pMgBpNa.*pNaBpCa - PhiK.*pCaBpH.*pHBpCa.*pMgBpNa.*pNaBpK - PhiK.*pCaBpH.*pHBpNa.*pMgBpK.*pNaBpCa + PhiK.*pCaBpH.*pHBpNa.*pMgBpCa.*pNaBpK + PhiK.*pCaBpK.*pHBpCa.*pMgBpNa.*pNaBpH + PhiK.*pCaBpK.*pHBpNa.*pMgBpH.*pNaBpCa - PhiK.*pCaBpK.*pHBpNa.*pMgBpCa.*pNaBpH - PhiK.*pCaBpNa.*pHBpK.*pMgBpH.*pNaBpCa + PhiK.*pCaBpNa.*pHBpK.*pMgBpCa.*pNaBpH + PhiK.*pCaBpNa.*pHBpCa.*pMgBpH.*pNaBpK - PhiK.*pCaBpNa.*pHBpCa.*pMgBpK.*pNaBpH + PhiH.*pCaBpH.*pKBpCa.*pMgBpNa.*pNaBpK + PhiH.*pCaBpH.*pKBpNa.*pMgBpK.*pNaBpCa - PhiH.*pCaBpH.*pKBpNa.*pMgBpCa.*pNaBpK + PhiH.*pCaBpK.*pKBpH.*pMgBpNa.*pNaBpCa - PhiH.*pCaBpK.*pKBpCa.*pMgBpNa.*pNaBpH - PhiH.*pCaBpK.*pKBpNa.*pMgBpH.*pNaBpCa + PhiH.*pCaBpK.*pKBpNa.*pMgBpCa.*pNaBpH - PhiH.*pCaBpNa.*pKBpH.*pMgBpK.*pNaBpCa + PhiH.*pCaBpNa.*pKBpH.*pMgBpCa.*pNaBpK - PhiH.*pCaBpNa.*pKBpCa.*pMgBpH.*pNaBpK + PhiH.*pCaBpNa.*pKBpCa.*pMgBpK.*pNaBpH - PhiCa.*pHBpK.*pKBpH.*pMgBpNa.*pNaBpCa + PhiCa.*pHBpK.*pKBpCa.*pMgBpNa.*pNaBpH + PhiCa.*pHBpK.*pKBpNa.*pMgBpH.*pNaBpCa - PhiCa.*pHBpK.*pKBpNa.*pMgBpCa.*pNaBpH + PhiCa.*pHBpCa.*pKBpH.*pMgBpNa.*pNaBpK - PhiCa.*pHBpCa.*pKBpNa.*pMgBpH.*pNaBpK + PhiCa.*pHBpCa.*pKBpNa.*pMgBpK.*pNaBpH + PhiCa.*pHBpNa.*pKBpH.*pMgBpK.*pNaBpCa - PhiCa.*pHBpNa.*pKBpH.*pMgBpCa.*pNaBpK + PhiCa.*pHBpNa.*pKBpCa.*pMgBpH.*pNaBpK - PhiCa.*pHBpNa.*pKBpCa.*pMgBpK.*pNaBpH + DCa.*DH.*DK.*DNa.*PhiMg - DCa.*DH.*DK.*PhiNa.*pMgBpNa - DCa.*DH.*DNa.*PhiK.*pMgBpK - DCa.*DK.*DNa.*PhiH.*pMgBpH - DH.*DK.*DNa.*PhiCa.*pMgBpCa)./(DH.*DK.*DMg.*pCaBpNa.*pNaBpCa + DH.*DK.*DNa.*pCaBpMg.*pMgBpCa + DCa.*DH.*DK.*pMgBpNa.*pNaBpMg + DK.*DMg.*DNa.*pCaBpH.*pHBpCa + DH.*DMg.*DNa.*pCaBpK.*pKBpCa + DCa.*DK.*DMg.*pHBpNa.*pNaBpH + DCa.*DH.*DMg.*pKBpNa.*pNaBpK + DCa.*DK.*DNa.*pHBpMg.*pMgBpH + DCa.*DH.*DNa.*pKBpMg.*pMgBpK + DCa.*DMg.*DNa.*pHBpK.*pKBpH - DH.*DK.*pCaBpNa.*pMgBpCa.*pNaBpMg - DH.*DK.*pCaBpMg.*pMgBpNa.*pNaBpCa - DK.*DMg.*pCaBpH.*pHBpNa.*pNaBpCa - DK.*DMg.*pCaBpNa.*pHBpCa.*pNaBpH - DH.*DMg.*pCaBpK.*pKBpNa.*pNaBpCa - DH.*DMg.*pCaBpNa.*pKBpCa.*pNaBpK - DK.*DNa.*pCaBpH.*pHBpMg.*pMgBpCa - DK.*DNa.*pCaBpMg.*pHBpCa.*pMgBpH - DH.*DNa.*pCaBpK.*pKBpMg.*pMgBpCa - DH.*DNa.*pCaBpMg.*pKBpCa.*pMgBpK - DCa.*DK.*pHBpNa.*pMgBpH.*pNaBpMg - DCa.*DK.*pHBpMg.*pMgBpNa.*pNaBpH - DCa.*DH.*pKBpNa.*pMgBpK.*pNaBpMg - DCa.*DH.*pKBpMg.*pMgBpNa.*pNaBpK - DMg.*DNa.*pCaBpH.*pHBpK.*pKBpCa - DMg.*DNa.*pCaBpK.*pHBpCa.*pKBpH - DCa.*DMg.*pHBpK.*pKBpNa.*pNaBpH - DCa.*DMg.*pHBpNa.*pKBpH.*pNaBpK - DCa.*DNa.*pHBpK.*pKBpMg.*pMgBpH - DCa.*DNa.*pHBpMg.*pKBpH.*pMgBpK - DK.*pCaBpH.*pHBpCa.*pMgBpNa.*pNaBpMg + DK.*pCaBpH.*pHBpNa.*pMgBpCa.*pNaBpMg + DK.*pCaBpH.*pHBpMg.*pMgBpNa.*pNaBpCa + DK.*pCaBpNa.*pHBpCa.*pMgBpH.*pNaBpMg - DK.*pCaBpNa.*pHBpMg.*pMgBpH.*pNaBpCa + DK.*pCaBpNa.*pHBpMg.*pMgBpCa.*pNaBpH + DK.*pCaBpMg.*pHBpCa.*pMgBpNa.*pNaBpH + DK.*pCaBpMg.*pHBpNa.*pMgBpH.*pNaBpCa - DK.*pCaBpMg.*pHBpNa.*pMgBpCa.*pNaBpH - DH.*pCaBpK.*pKBpCa.*pMgBpNa.*pNaBpMg + DH.*pCaBpK.*pKBpNa.*pMgBpCa.*pNaBpMg + DH.*pCaBpK.*pKBpMg.*pMgBpNa.*pNaBpCa + DH.*pCaBpNa.*pKBpCa.*pMgBpK.*pNaBpMg - DH.*pCaBpNa.*pKBpMg.*pMgBpK.*pNaBpCa + DH.*pCaBpNa.*pKBpMg.*pMgBpCa.*pNaBpK + DH.*pCaBpMg.*pKBpCa.*pMgBpNa.*pNaBpK + DH.*pCaBpMg.*pKBpNa.*pMgBpK.*pNaBpCa - DH.*pCaBpMg.*pKBpNa.*pMgBpCa.*pNaBpK + DMg.*pCaBpH.*pHBpK.*pKBpNa.*pNaBpCa - DMg.*pCaBpH.*pHBpCa.*pKBpNa.*pNaBpK + DMg.*pCaBpH.*pHBpNa.*pKBpCa.*pNaBpK + DMg.*pCaBpK.*pHBpCa.*pKBpNa.*pNaBpH + DMg.*pCaBpK.*pHBpNa.*pKBpH.*pNaBpCa - DMg.*pCaBpK.*pHBpNa.*pKBpCa.*pNaBpH - DMg.*pCaBpNa.*pHBpK.*pKBpH.*pNaBpCa + DMg.*pCaBpNa.*pHBpK.*pKBpCa.*pNaBpH + DMg.*pCaBpNa.*pHBpCa.*pKBpH.*pNaBpK + DNa.*pCaBpH.*pHBpK.*pKBpMg.*pMgBpCa - DNa.*pCaBpH.*pHBpCa.*pKBpMg.*pMgBpK + DNa.*pCaBpH.*pHBpMg.*pKBpCa.*pMgBpK + DNa.*pCaBpK.*pHBpCa.*pKBpMg.*pMgBpH + DNa.*pCaBpK.*pHBpMg.*pKBpH.*pMgBpCa - DNa.*pCaBpK.*pHBpMg.*pKBpCa.*pMgBpH - DNa.*pCaBpMg.*pHBpK.*pKBpH.*pMgBpCa + DNa.*pCaBpMg.*pHBpK.*pKBpCa.*pMgBpH + DNa.*pCaBpMg.*pHBpCa.*pKBpH.*pMgBpK - DCa.*pHBpK.*pKBpH.*pMgBpNa.*pNaBpMg + DCa.*pHBpK.*pKBpNa.*pMgBpH.*pNaBpMg + DCa.*pHBpK.*pKBpMg.*pMgBpNa.*pNaBpH + DCa.*pHBpNa.*pKBpH.*pMgBpK.*pNaBpMg + DCa.*pHBpNa.*pKBpMg.*pMgBpH.*pNaBpK - DCa.*pHBpNa.*pKBpMg.*pMgBpK.*pNaBpH + DCa.*pHBpMg.*pKBpH.*pMgBpNa.*pNaBpK - DCa.*pHBpMg.*pKBpNa.*pMgBpH.*pNaBpK + DCa.*pHBpMg.*pKBpNa.*pMgBpK.*pNaBpH + pCaBpH.*pHBpK.*pKBpCa.*pMgBpNa.*pNaBpMg - pCaBpH.*pHBpK.*pKBpNa.*pMgBpCa.*pNaBpMg - pCaBpH.*pHBpK.*pKBpMg.*pMgBpNa.*pNaBpCa + pCaBpH.*pHBpCa.*pKBpNa.*pMgBpK.*pNaBpMg + pCaBpH.*pHBpCa.*pKBpMg.*pMgBpNa.*pNaBpK - pCaBpH.*pHBpNa.*pKBpCa.*pMgBpK.*pNaBpMg + pCaBpH.*pHBpNa.*pKBpMg.*pMgBpK.*pNaBpCa - pCaBpH.*pHBpNa.*pKBpMg.*pMgBpCa.*pNaBpK - pCaBpH.*pHBpMg.*pKBpCa.*pMgBpNa.*pNaBpK - pCaBpH.*pHBpMg.*pKBpNa.*pMgBpK.*pNaBpCa + pCaBpH.*pHBpMg.*pKBpNa.*pMgBpCa.*pNaBpK + pCaBpK.*pHBpCa.*pKBpH.*pMgBpNa.*pNaBpMg - pCaBpK.*pHBpCa.*pKBpNa.*pMgBpH.*pNaBpMg - pCaBpK.*pHBpCa.*pKBpMg.*pMgBpNa.*pNaBpH - pCaBpK.*pHBpNa.*pKBpH.*pMgBpCa.*pNaBpMg + pCaBpK.*pHBpNa.*pKBpCa.*pMgBpH.*pNaBpMg - pCaBpK.*pHBpNa.*pKBpMg.*pMgBpH.*pNaBpCa + pCaBpK.*pHBpNa.*pKBpMg.*pMgBpCa.*pNaBpH - pCaBpK.*pHBpMg.*pKBpH.*pMgBpNa.*pNaBpCa + pCaBpK.*pHBpMg.*pKBpCa.*pMgBpNa.*pNaBpH + pCaBpK.*pHBpMg.*pKBpNa.*pMgBpH.*pNaBpCa - pCaBpK.*pHBpMg.*pKBpNa.*pMgBpCa.*pNaBpH + pCaBpNa.*pHBpK.*pKBpH.*pMgBpCa.*pNaBpMg - pCaBpNa.*pHBpK.*pKBpCa.*pMgBpH.*pNaBpMg + pCaBpNa.*pHBpK.*pKBpMg.*pMgBpH.*pNaBpCa - pCaBpNa.*pHBpK.*pKBpMg.*pMgBpCa.*pNaBpH - pCaBpNa.*pHBpCa.*pKBpH.*pMgBpK.*pNaBpMg - pCaBpNa.*pHBpCa.*pKBpMg.*pMgBpH.*pNaBpK + pCaBpNa.*pHBpCa.*pKBpMg.*pMgBpK.*pNaBpH + pCaBpNa.*pHBpMg.*pKBpH.*pMgBpK.*pNaBpCa - pCaBpNa.*pHBpMg.*pKBpH.*pMgBpCa.*pNaBpK + pCaBpNa.*pHBpMg.*pKBpCa.*pMgBpH.*pNaBpK - pCaBpNa.*pHBpMg.*pKBpCa.*pMgBpK.*pNaBpH + pCaBpMg.*pHBpK.*pKBpH.*pMgBpNa.*pNaBpCa - pCaBpMg.*pHBpK.*pKBpCa.*pMgBpNa.*pNaBpH - pCaBpMg.*pHBpK.*pKBpNa.*pMgBpH.*pNaBpCa + pCaBpMg.*pHBpK.*pKBpNa.*pMgBpCa.*pNaBpH - pCaBpMg.*pHBpCa.*pKBpH.*pMgBpNa.*pNaBpK + pCaBpMg.*pHBpCa.*pKBpNa.*pMgBpH.*pNaBpK - pCaBpMg.*pHBpCa.*pKBpNa.*pMgBpK.*pNaBpH - pCaBpMg.*pHBpNa.*pKBpH.*pMgBpK.*pNaBpCa + pCaBpMg.*pHBpNa.*pKBpH.*pMgBpCa.*pNaBpK - pCaBpMg.*pHBpNa.*pKBpCa.*pMgBpH.*pNaBpK + pCaBpMg.*pHBpNa.*pKBpCa.*pMgBpK.*pNaBpH - DCa.*DH.*DK.*DMg.*DNa);
dCadt = -(DH.*DK.*PhiNa.*pCaBpMg.*pMgBpNa + DK.*DMg.*PhiNa.*pCaBpH.*pHBpNa + DH.*DMg.*PhiNa.*pCaBpK.*pKBpNa + DH.*DK.*PhiMg.*pCaBpNa.*pNaBpMg + DK.*DNa.*PhiMg.*pCaBpH.*pHBpMg + DH.*DNa.*PhiMg.*pCaBpK.*pKBpMg + DH.*DMg.*PhiK.*pCaBpNa.*pNaBpK + DH.*DNa.*PhiK.*pCaBpMg.*pMgBpK + DMg.*DNa.*PhiK.*pCaBpH.*pHBpK + DK.*DMg.*PhiH.*pCaBpNa.*pNaBpH + DK.*DNa.*PhiH.*pCaBpMg.*pMgBpH + DMg.*DNa.*PhiH.*pCaBpK.*pKBpH - DH.*DK.*PhiCa.*pMgBpNa.*pNaBpMg - DK.*DMg.*PhiCa.*pHBpNa.*pNaBpH - DH.*DMg.*PhiCa.*pKBpNa.*pNaBpK - DK.*DNa.*PhiCa.*pHBpMg.*pMgBpH - DH.*DNa.*PhiCa.*pKBpMg.*pMgBpK - DMg.*DNa.*PhiCa.*pHBpK.*pKBpH - DK.*PhiNa.*pCaBpH.*pHBpMg.*pMgBpNa + DK.*PhiNa.*pCaBpNa.*pHBpMg.*pMgBpH - DK.*PhiNa.*pCaBpMg.*pHBpNa.*pMgBpH - DH.*PhiNa.*pCaBpK.*pKBpMg.*pMgBpNa + DH.*PhiNa.*pCaBpNa.*pKBpMg.*pMgBpK - DH.*PhiNa.*pCaBpMg.*pKBpNa.*pMgBpK - DMg.*PhiNa.*pCaBpH.*pHBpK.*pKBpNa - DMg.*PhiNa.*pCaBpK.*pHBpNa.*pKBpH + DMg.*PhiNa.*pCaBpNa.*pHBpK.*pKBpH - DK.*PhiMg.*pCaBpH.*pHBpNa.*pNaBpMg - DK.*PhiMg.*pCaBpNa.*pHBpMg.*pNaBpH + DK.*PhiMg.*pCaBpMg.*pHBpNa.*pNaBpH - DH.*PhiMg.*pCaBpK.*pKBpNa.*pNaBpMg - DH.*PhiMg.*pCaBpNa.*pKBpMg.*pNaBpK + DH.*PhiMg.*pCaBpMg.*pKBpNa.*pNaBpK - DNa.*PhiMg.*pCaBpH.*pHBpK.*pKBpMg - DNa.*PhiMg.*pCaBpK.*pHBpMg.*pKBpH + DNa.*PhiMg.*pCaBpMg.*pHBpK.*pKBpH + DH.*PhiK.*pCaBpK.*pMgBpNa.*pNaBpMg - DH.*PhiK.*pCaBpNa.*pMgBpK.*pNaBpMg - DH.*PhiK.*pCaBpMg.*pMgBpNa.*pNaBpK - DMg.*PhiK.*pCaBpH.*pHBpNa.*pNaBpK + DMg.*PhiK.*pCaBpK.*pHBpNa.*pNaBpH - DMg.*PhiK.*pCaBpNa.*pHBpK.*pNaBpH - DNa.*PhiK.*pCaBpH.*pHBpMg.*pMgBpK + DNa.*PhiK.*pCaBpK.*pHBpMg.*pMgBpH - DNa.*PhiK.*pCaBpMg.*pHBpK.*pMgBpH + DK.*PhiH.*pCaBpH.*pMgBpNa.*pNaBpMg - DK.*PhiH.*pCaBpNa.*pMgBpH.*pNaBpMg - DK.*PhiH.*pCaBpMg.*pMgBpNa.*pNaBpH + DMg.*PhiH.*pCaBpH.*pKBpNa.*pNaBpK - DMg.*PhiH.*pCaBpK.*pKBpNa.*pNaBpH - DMg.*PhiH.*pCaBpNa.*pKBpH.*pNaBpK + DNa.*PhiH.*pCaBpH.*pKBpMg.*pMgBpK - DNa.*PhiH.*pCaBpK.*pKBpMg.*pMgBpH - DNa.*PhiH.*pCaBpMg.*pKBpH.*pMgBpK + DK.*PhiCa.*pHBpNa.*pMgBpH.*pNaBpMg + DK.*PhiCa.*pHBpMg.*pMgBpNa.*pNaBpH + DH.*PhiCa.*pKBpNa.*pMgBpK.*pNaBpMg + DH.*PhiCa.*pKBpMg.*pMgBpNa.*pNaBpK + DMg.*PhiCa.*pHBpK.*pKBpNa.*pNaBpH + DMg.*PhiCa.*pHBpNa.*pKBpH.*pNaBpK + DNa.*PhiCa.*pHBpK.*pKBpMg.*pMgBpH + DNa.*PhiCa.*pHBpMg.*pKBpH.*pMgBpK + PhiNa.*pCaBpH.*pHBpK.*pKBpMg.*pMgBpNa - PhiNa.*pCaBpH.*pHBpNa.*pKBpMg.*pMgBpK + PhiNa.*pCaBpH.*pHBpMg.*pKBpNa.*pMgBpK + PhiNa.*pCaBpK.*pHBpNa.*pKBpMg.*pMgBpH + PhiNa.*pCaBpK.*pHBpMg.*pKBpH.*pMgBpNa - PhiNa.*pCaBpK.*pHBpMg.*pKBpNa.*pMgBpH - PhiNa.*pCaBpNa.*pHBpK.*pKBpMg.*pMgBpH - PhiNa.*pCaBpNa.*pHBpMg.*pKBpH.*pMgBpK - PhiNa.*pCaBpMg.*pHBpK.*pKBpH.*pMgBpNa + PhiNa.*pCaBpMg.*pHBpK.*pKBpNa.*pMgBpH + PhiNa.*pCaBpMg.*pHBpNa.*pKBpH.*pMgBpK + PhiMg.*pCaBpH.*pHBpK.*pKBpNa.*pNaBpMg + PhiMg.*pCaBpH.*pHBpNa.*pKBpMg.*pNaBpK - PhiMg.*pCaBpH.*pHBpMg.*pKBpNa.*pNaBpK + PhiMg.*pCaBpK.*pHBpNa.*pKBpH.*pNaBpMg - PhiMg.*pCaBpK.*pHBpNa.*pKBpMg.*pNaBpH + PhiMg.*pCaBpK.*pHBpMg.*pKBpNa.*pNaBpH - PhiMg.*pCaBpNa.*pHBpK.*pKBpH.*pNaBpMg + PhiMg.*pCaBpNa.*pHBpK.*pKBpMg.*pNaBpH + PhiMg.*pCaBpNa.*pHBpMg.*pKBpH.*pNaBpK - PhiMg.*pCaBpMg.*pHBpK.*pKBpNa.*pNaBpH - PhiMg.*pCaBpMg.*pHBpNa.*pKBpH.*pNaBpK - PhiK.*pCaBpH.*pHBpK.*pMgBpNa.*pNaBpMg + PhiK.*pCaBpH.*pHBpNa.*pMgBpK.*pNaBpMg + PhiK.*pCaBpH.*pHBpMg.*pMgBpNa.*pNaBpK - PhiK.*pCaBpK.*pHBpNa.*pMgBpH.*pNaBpMg - PhiK.*pCaBpK.*pHBpMg.*pMgBpNa.*pNaBpH + PhiK.*pCaBpNa.*pHBpK.*pMgBpH.*pNaBpMg - PhiK.*pCaBpNa.*pHBpMg.*pMgBpH.*pNaBpK + PhiK.*pCaBpNa.*pHBpMg.*pMgBpK.*pNaBpH + PhiK.*pCaBpMg.*pHBpK.*pMgBpNa.*pNaBpH + PhiK.*pCaBpMg.*pHBpNa.*pMgBpH.*pNaBpK - PhiK.*pCaBpMg.*pHBpNa.*pMgBpK.*pNaBpH - PhiH.*pCaBpH.*pKBpNa.*pMgBpK.*pNaBpMg - PhiH.*pCaBpH.*pKBpMg.*pMgBpNa.*pNaBpK - PhiH.*pCaBpK.*pKBpH.*pMgBpNa.*pNaBpMg + PhiH.*pCaBpK.*pKBpNa.*pMgBpH.*pNaBpMg + PhiH.*pCaBpK.*pKBpMg.*pMgBpNa.*pNaBpH + PhiH.*pCaBpNa.*pKBpH.*pMgBpK.*pNaBpMg + PhiH.*pCaBpNa.*pKBpMg.*pMgBpH.*pNaBpK - PhiH.*pCaBpNa.*pKBpMg.*pMgBpK.*pNaBpH + PhiH.*pCaBpMg.*pKBpH.*pMgBpNa.*pNaBpK - PhiH.*pCaBpMg.*pKBpNa.*pMgBpH.*pNaBpK + PhiH.*pCaBpMg.*pKBpNa.*pMgBpK.*pNaBpH + PhiCa.*pHBpK.*pKBpH.*pMgBpNa.*pNaBpMg - PhiCa.*pHBpK.*pKBpNa.*pMgBpH.*pNaBpMg - PhiCa.*pHBpK.*pKBpMg.*pMgBpNa.*pNaBpH - PhiCa.*pHBpNa.*pKBpH.*pMgBpK.*pNaBpMg - PhiCa.*pHBpNa.*pKBpMg.*pMgBpH.*pNaBpK + PhiCa.*pHBpNa.*pKBpMg.*pMgBpK.*pNaBpH - PhiCa.*pHBpMg.*pKBpH.*pMgBpNa.*pNaBpK + PhiCa.*pHBpMg.*pKBpNa.*pMgBpH.*pNaBpK - PhiCa.*pHBpMg.*pKBpNa.*pMgBpK.*pNaBpH + DH.*DK.*DMg.*DNa.*PhiCa - DH.*DK.*DMg.*PhiNa.*pCaBpNa - DH.*DK.*DNa.*PhiMg.*pCaBpMg - DH.*DMg.*DNa.*PhiK.*pCaBpK - DK.*DMg.*DNa.*PhiH.*pCaBpH)./(DH.*DK.*DMg.*pCaBpNa.*pNaBpCa + DH.*DK.*DNa.*pCaBpMg.*pMgBpCa + DCa.*DH.*DK.*pMgBpNa.*pNaBpMg + DK.*DMg.*DNa.*pCaBpH.*pHBpCa + DH.*DMg.*DNa.*pCaBpK.*pKBpCa + DCa.*DK.*DMg.*pHBpNa.*pNaBpH + DCa.*DH.*DMg.*pKBpNa.*pNaBpK + DCa.*DK.*DNa.*pHBpMg.*pMgBpH + DCa.*DH.*DNa.*pKBpMg.*pMgBpK + DCa.*DMg.*DNa.*pHBpK.*pKBpH - DH.*DK.*pCaBpNa.*pMgBpCa.*pNaBpMg - DH.*DK.*pCaBpMg.*pMgBpNa.*pNaBpCa - DK.*DMg.*pCaBpH.*pHBpNa.*pNaBpCa - DK.*DMg.*pCaBpNa.*pHBpCa.*pNaBpH - DH.*DMg.*pCaBpK.*pKBpNa.*pNaBpCa - DH.*DMg.*pCaBpNa.*pKBpCa.*pNaBpK - DK.*DNa.*pCaBpH.*pHBpMg.*pMgBpCa - DK.*DNa.*pCaBpMg.*pHBpCa.*pMgBpH - DH.*DNa.*pCaBpK.*pKBpMg.*pMgBpCa - DH.*DNa.*pCaBpMg.*pKBpCa.*pMgBpK - DCa.*DK.*pHBpNa.*pMgBpH.*pNaBpMg - DCa.*DK.*pHBpMg.*pMgBpNa.*pNaBpH - DCa.*DH.*pKBpNa.*pMgBpK.*pNaBpMg - DCa.*DH.*pKBpMg.*pMgBpNa.*pNaBpK - DMg.*DNa.*pCaBpH.*pHBpK.*pKBpCa - DMg.*DNa.*pCaBpK.*pHBpCa.*pKBpH - DCa.*DMg.*pHBpK.*pKBpNa.*pNaBpH - DCa.*DMg.*pHBpNa.*pKBpH.*pNaBpK - DCa.*DNa.*pHBpK.*pKBpMg.*pMgBpH - DCa.*DNa.*pHBpMg.*pKBpH.*pMgBpK - DK.*pCaBpH.*pHBpCa.*pMgBpNa.*pNaBpMg + DK.*pCaBpH.*pHBpNa.*pMgBpCa.*pNaBpMg + DK.*pCaBpH.*pHBpMg.*pMgBpNa.*pNaBpCa + DK.*pCaBpNa.*pHBpCa.*pMgBpH.*pNaBpMg - DK.*pCaBpNa.*pHBpMg.*pMgBpH.*pNaBpCa + DK.*pCaBpNa.*pHBpMg.*pMgBpCa.*pNaBpH + DK.*pCaBpMg.*pHBpCa.*pMgBpNa.*pNaBpH + DK.*pCaBpMg.*pHBpNa.*pMgBpH.*pNaBpCa - DK.*pCaBpMg.*pHBpNa.*pMgBpCa.*pNaBpH - DH.*pCaBpK.*pKBpCa.*pMgBpNa.*pNaBpMg + DH.*pCaBpK.*pKBpNa.*pMgBpCa.*pNaBpMg + DH.*pCaBpK.*pKBpMg.*pMgBpNa.*pNaBpCa + DH.*pCaBpNa.*pKBpCa.*pMgBpK.*pNaBpMg - DH.*pCaBpNa.*pKBpMg.*pMgBpK.*pNaBpCa + DH.*pCaBpNa.*pKBpMg.*pMgBpCa.*pNaBpK + DH.*pCaBpMg.*pKBpCa.*pMgBpNa.*pNaBpK + DH.*pCaBpMg.*pKBpNa.*pMgBpK.*pNaBpCa - DH.*pCaBpMg.*pKBpNa.*pMgBpCa.*pNaBpK + DMg.*pCaBpH.*pHBpK.*pKBpNa.*pNaBpCa - DMg.*pCaBpH.*pHBpCa.*pKBpNa.*pNaBpK + DMg.*pCaBpH.*pHBpNa.*pKBpCa.*pNaBpK + DMg.*pCaBpK.*pHBpCa.*pKBpNa.*pNaBpH + DMg.*pCaBpK.*pHBpNa.*pKBpH.*pNaBpCa - DMg.*pCaBpK.*pHBpNa.*pKBpCa.*pNaBpH - DMg.*pCaBpNa.*pHBpK.*pKBpH.*pNaBpCa + DMg.*pCaBpNa.*pHBpK.*pKBpCa.*pNaBpH + DMg.*pCaBpNa.*pHBpCa.*pKBpH.*pNaBpK + DNa.*pCaBpH.*pHBpK.*pKBpMg.*pMgBpCa - DNa.*pCaBpH.*pHBpCa.*pKBpMg.*pMgBpK + DNa.*pCaBpH.*pHBpMg.*pKBpCa.*pMgBpK + DNa.*pCaBpK.*pHBpCa.*pKBpMg.*pMgBpH + DNa.*pCaBpK.*pHBpMg.*pKBpH.*pMgBpCa - DNa.*pCaBpK.*pHBpMg.*pKBpCa.*pMgBpH - DNa.*pCaBpMg.*pHBpK.*pKBpH.*pMgBpCa + DNa.*pCaBpMg.*pHBpK.*pKBpCa.*pMgBpH + DNa.*pCaBpMg.*pHBpCa.*pKBpH.*pMgBpK - DCa.*pHBpK.*pKBpH.*pMgBpNa.*pNaBpMg + DCa.*pHBpK.*pKBpNa.*pMgBpH.*pNaBpMg + DCa.*pHBpK.*pKBpMg.*pMgBpNa.*pNaBpH + DCa.*pHBpNa.*pKBpH.*pMgBpK.*pNaBpMg + DCa.*pHBpNa.*pKBpMg.*pMgBpH.*pNaBpK - DCa.*pHBpNa.*pKBpMg.*pMgBpK.*pNaBpH + DCa.*pHBpMg.*pKBpH.*pMgBpNa.*pNaBpK - DCa.*pHBpMg.*pKBpNa.*pMgBpH.*pNaBpK + DCa.*pHBpMg.*pKBpNa.*pMgBpK.*pNaBpH + pCaBpH.*pHBpK.*pKBpCa.*pMgBpNa.*pNaBpMg - pCaBpH.*pHBpK.*pKBpNa.*pMgBpCa.*pNaBpMg - pCaBpH.*pHBpK.*pKBpMg.*pMgBpNa.*pNaBpCa + pCaBpH.*pHBpCa.*pKBpNa.*pMgBpK.*pNaBpMg + pCaBpH.*pHBpCa.*pKBpMg.*pMgBpNa.*pNaBpK - pCaBpH.*pHBpNa.*pKBpCa.*pMgBpK.*pNaBpMg + pCaBpH.*pHBpNa.*pKBpMg.*pMgBpK.*pNaBpCa - pCaBpH.*pHBpNa.*pKBpMg.*pMgBpCa.*pNaBpK - pCaBpH.*pHBpMg.*pKBpCa.*pMgBpNa.*pNaBpK - pCaBpH.*pHBpMg.*pKBpNa.*pMgBpK.*pNaBpCa + pCaBpH.*pHBpMg.*pKBpNa.*pMgBpCa.*pNaBpK + pCaBpK.*pHBpCa.*pKBpH.*pMgBpNa.*pNaBpMg - pCaBpK.*pHBpCa.*pKBpNa.*pMgBpH.*pNaBpMg - pCaBpK.*pHBpCa.*pKBpMg.*pMgBpNa.*pNaBpH - pCaBpK.*pHBpNa.*pKBpH.*pMgBpCa.*pNaBpMg + pCaBpK.*pHBpNa.*pKBpCa.*pMgBpH.*pNaBpMg - pCaBpK.*pHBpNa.*pKBpMg.*pMgBpH.*pNaBpCa + pCaBpK.*pHBpNa.*pKBpMg.*pMgBpCa.*pNaBpH - pCaBpK.*pHBpMg.*pKBpH.*pMgBpNa.*pNaBpCa + pCaBpK.*pHBpMg.*pKBpCa.*pMgBpNa.*pNaBpH + pCaBpK.*pHBpMg.*pKBpNa.*pMgBpH.*pNaBpCa - pCaBpK.*pHBpMg.*pKBpNa.*pMgBpCa.*pNaBpH + pCaBpNa.*pHBpK.*pKBpH.*pMgBpCa.*pNaBpMg - pCaBpNa.*pHBpK.*pKBpCa.*pMgBpH.*pNaBpMg + pCaBpNa.*pHBpK.*pKBpMg.*pMgBpH.*pNaBpCa - pCaBpNa.*pHBpK.*pKBpMg.*pMgBpCa.*pNaBpH - pCaBpNa.*pHBpCa.*pKBpH.*pMgBpK.*pNaBpMg - pCaBpNa.*pHBpCa.*pKBpMg.*pMgBpH.*pNaBpK + pCaBpNa.*pHBpCa.*pKBpMg.*pMgBpK.*pNaBpH + pCaBpNa.*pHBpMg.*pKBpH.*pMgBpK.*pNaBpCa - pCaBpNa.*pHBpMg.*pKBpH.*pMgBpCa.*pNaBpK + pCaBpNa.*pHBpMg.*pKBpCa.*pMgBpH.*pNaBpK - pCaBpNa.*pHBpMg.*pKBpCa.*pMgBpK.*pNaBpH + pCaBpMg.*pHBpK.*pKBpH.*pMgBpNa.*pNaBpCa - pCaBpMg.*pHBpK.*pKBpCa.*pMgBpNa.*pNaBpH - pCaBpMg.*pHBpK.*pKBpNa.*pMgBpH.*pNaBpCa + pCaBpMg.*pHBpK.*pKBpNa.*pMgBpCa.*pNaBpH - pCaBpMg.*pHBpCa.*pKBpH.*pMgBpNa.*pNaBpK + pCaBpMg.*pHBpCa.*pKBpNa.*pMgBpH.*pNaBpK - pCaBpMg.*pHBpCa.*pKBpNa.*pMgBpK.*pNaBpH - pCaBpMg.*pHBpNa.*pKBpH.*pMgBpK.*pNaBpCa + pCaBpMg.*pHBpNa.*pKBpH.*pMgBpCa.*pNaBpK - pCaBpMg.*pHBpNa.*pKBpCa.*pMgBpH.*pNaBpK + pCaBpMg.*pHBpNa.*pKBpCa.*pMgBpK.*pNaBpH - DCa.*DH.*DK.*DMg.*DNa);

% M/s b/c Phi
f(p.i.H_c,:) = dHdt;
f(p.i.K_c,:) = dKdt;
f(p.i.Na_c,:) = dNadt;
f(p.i.Mg_c,:) = dMgdt;
f(p.i.Ca_c,:) = dCadt;
end