###############################################
# Functions for generating rainfall scenarios #
# Mary Ollenburger                            #
# mollenburger@gmail.com                      #
# July 2012                                   #
# For current version contact the author	  #
###############################################

require(plyr)
# import metfile for testing: change the file path to wherever yours is located.
# All functions work with APSIM metfiles: files in general should not have missing data, and should have columns
# with names "year","doy", and "rain"
met1<-read.table('zombwe.met',skip=9)
names(met1)<-c('year','doy','radn','maxt','mint','rain')
#add date and month columns for later calculations
met1$date<-as.Date(paste(met1$year,met1$doy,sep='-'),format='%Y-%j')
met1$month<-as.numeric(strftime(met1$date,'%m'))
met1$Hyear<-ifelse(met1$month %in% 10:12,met1$year+1,met1$year)

# Because the plyr package makes applying this over specific years easy the function is written for 
# a 1-year dataset. Creating a harvest year (Hyear) column in the met file allows us to start the process
# at any point in the year

####################
# rainfall removal #
####################

# change rainfall amount by removing a percentage (PctReduce) from one in countN rainy days.
ChangeRainAmt<-function(met,countN,startCount,Threshold,PctReduce){
	met$count<-0
	met$count[1]<-startCount
	met$rainOut<-met$rain
	for(i in 2:nrow(met)){
		ifelse(met$rain[i]>Threshold, met$count[i]<-met$count[i-1]+1,met$count[i]<-met$count[i-1])
		if(met$count[i]%%countN==0) met$rainOut[i]<-met$rain[i]*(100-PctReduce)/100
		}
	met$rainReduced<-met$rain-met$rainOut
	met
		}

#function can be used continuously over the rainfall record or by year using plyr:		
dayOut<-ddply(met1,.(year),function(df)ChangeRainAmt(df,5,0,0.85,100))

# change rainfall amount by removing a percentage (PctReduce) from one in countN rainy spells: spells can be of any
# duration >=1.

ChangeSpell<-function(met,countN,startCount,Threshold,PctReduce){
	met$rainlag<-c(NA,met$rain[1:(nrow(met)-1)])
	met$countspell<-0
	met$countspell[1]<-startCount
	met$rainOut<-met$rain
	for(i in 2:nrow(met)){
		if(met$rain[i]>Threshold & met$rainlag[i]<Threshold & !is.na(met$rainlag[i])){
			met$countspell[i]<-met$countspell[i-1]+1
			}else{met$countspell[i]<-met$countspell[i-1]}
		if(met$countspell[i]%%countN==0) met$rainOut[i]<-met$rain[i]*(100-PctReduce)/100
		}
		met$rainReduced<-met$rain-met$rainOut
		met
		}	

###########################
# rainfall redistribution #
###########################

# result of either of the previous 2 functions can be input to redistribute.[...]() 
# redistribute functions distribute a percentage of the rainfall previously removed in order to reach
# GoalOut, goal reduction fraction of baseline rainfall (e.g. for 10% reduction use 0.1)
# If the total to be redistributed is less than zero, redistribution will result in some days with negative rainfall

#redistribute rainfall amount removed over all remaining rainy days
Redistribute.raindays<-function(out,Threshold,GoalOut){
	raindays<-sum(out$rainOut>Threshold)
	totRedistributed<-(1-GoalOut)*sum(out$rain)-sum(out$rainOut)
	add<-totRedistributed/raindays
	out$rainFinal<-out$rainOut
	for(i in 1:nrow(out)){
		if(out$rainFinal[i]>Threshold) out$rainFinal[i]<-out$rainOut[i]+add}
	if(totRedistributed<0) {
				warning('warning: negative rain redistributed: may not exactly match goal')
				return(out)
				}else{
					return(out)
	}}

# redistribute as above, on 1 in ndays calendar days
Redistribute.ndays<-function(out, ndays,GoalOut){
		totRedistributed<-(1-GoalOut)*sum(out$rain)-sum(out$rainOut)
		days<-nrow(out)/ndays
		add<-totRedistributed/days
		out$rainFinal<-out$rainOut
		#can't have negative rain, so if totRedistributed is negative, this won't give you the goal
		for(i in 1:nrow(out)){if (i%%ndays==0) out$rainFinal[i]<-max(0,out$rainOut[i]+add)}
		if(totRedistributed<0) {
				warning('warning: negative rain redistributed: may not exactly match goal')
				return(out)
				}else{
					return(out)
	}}

# redistribute as above, on 1 in ndays remaining rainy days
Redistribute.Nrain<-function(out,Threshold,ndays,GoalOut){	
		totRedistributed<-(1-GoalOut)*sum(out$rain)-sum(out$rainOut)
		raindays<-sum(out$rainOut>Threshold)
		days<-raindays/ndays
		add<-totRedistributed/days
		out$rainFinal<-out$rainOut
		out$countOut<-0
		for(i in 2:nrow(out)){
			ifelse(out$rainOut[i]>Threshold, out$countOut[i]<-out$countOut[i-1]+1,out$countOut[i]<-out$countOut[i-1])
			if(out$countOut[i]%%ndays==0 & out$countOut[i-1]!=out$countOut[i] & out$countOut[i]!=0) max(0,out$rainFinal[i]<-(out$rainOut[i]+add))}
			if(totRedistributed<0) {
				warning('warning: negative rain redistributed: may not exactly match goal')
				return(out)
				}else{
					return(out)
			}}


#add a specific amount to baseline rainfall on 1 in ndays rainy days
add.nrain<-function(met,Threshold,ndays,GoalOut){	
		totRedistributed<-GoalOut*sum(met$rain)
		raindays<-sum(met$rain>Threshold)
		days<-raindays/ndays
		add<-totRedistributed/days
		met$rainOut<-met$rain
		met$count<-0
		for(i in 2:nrow(met)){
			ifelse(met$rain[i]>Threshold, met$count[i]<-met$count[i-1]+1,met$count[i]<-met$count[i-1])
			if(met$count[i]%%ndays==0 & met$count[i-1]!=met$count[i] & met$count[i]!=0) met$rainOut[i]<-(met$rain[i]+add)}
			met}

# these should be used with plyr to ensure each year has the appropriate (goalOut) rainfall reduction.
# they will display warnings when years may not exactly match GoalOut.
redistOut<-ddply(dayOut,.(year),function(df) Redistribute.raindays(df,0.85,0.1))
redistNROut<-ddply(dayOut,.(year),function(df) Redistribute.Nrain(df,0.85,5,0.1))
redistNout<-ddply(dayOut,.(year),function(df) Redistribute.ndays(df,5,0.1))

########################################
# removal restricted to calendar dates #
########################################

# Remove rainfall from one in n rainy days for "totdays" rainy days FOLLOWING a calendar date "startdate" 
# startdate must be as julian day or "day of year"
ChangeRain.start<-function(met,countN,Threshold,PctReduce,startdate,totdays){
	met$count<-0
	met$rainOut<-met$rain
	for(i in 2:nrow(met)){
		if(met$doy[i]>startdate){
		ifelse(met$rain[i]>Threshold, met$count[i]<-met$count[i-1]+1,met$count[i]<-met$count[i-1])
		if(met$count[i]%%countN==0&met$count[i]<=totdays) met$rainOut[i]<-met$rain[i]*(100-PctReduce)/100
		}}
	met$rainReduced<-met$rain-met$rainOut
	return(met)}

start.removeN<-ddply(met1,.(Hyear),function(df) ChangeRain.start(df,1,0.85,100,300,10))

# Remove rainfall from one in countN rainy days for a calendar dates between "startdate" and "enddate" 
# specified as julian days (day of year)
ChangeRain.cal<-function(met,countN,Threshold,PctReduce,startdate,enddate){
	met$count<-0
	met$rainOut<-met$rain
	for(i in 2:nrow(met)){
		if(met$doy[i]>startdate&met$doy[i]<enddate){
		ifelse(met$rain[i]>Threshold, met$count[i]<-met$count[i-1]+1,met$count[i]<-met$count[i-1])
		if(met$count[i]%%countN==0) met$rainOut[i]<-met$rain[i]*(100-PctReduce)/100
		}}
	met$rainReduced<-met$rain-met$rainOut
	return(met)}

start.remove.cal<-ddply(met1,.(year),function(df) ChangeRain.cal(df,2,0.85,100,244,334))

# redistribution in a calendar date range: on 1 in ndays rainy days between "startdate" and "enddate"
Redistribute.cal<-function(out,Threshold,ndays,GoalOut,startdate,enddate){	
		totRedistributed<-(1-GoalOut)*sum(out$rain)-sum(out$rainOut)
			work<-out[out$doy %in% startdate:enddate,]
			raindays<-sum(work$rainOut>Threshold)
			days<-raindays/ndays
			add<-totRedistributed/days
			out$rainFinal<-out$rainOut
			out$countOut<-0
		for(i in 2:nrow(out)){
			if(out$doy[i]>startdate & out$doy<enddate){
			ifelse(out$rainOut[i]>Threshold, out$countOut[i]<-out$countOut[i-1]+1,out$countOut[i]<-out$countOut[i-1])
			if(out$countOut[i]%%ndays==0 & out$countOut[i-1]!=out$countOut[i] & out$countOut[i]!=0) max(0,out$rainFinal[i]<-(out$rainOut[i]+add))}}
				if(totRedistributed<0) {
				warning('warning: negative rain redistributed: may not exactly match goal')
				return(out)
				}else{
					return(out)
			}}

redist.cal<-ddply(start.remove.cal,.(year),function(df) Redistribute.cal(df,0.85,1,0.25,244,334))

####################################################################
# Removal and redistribution based on start or end of rainy season #
####################################################################

# find start of rainy season: first instance of rainfall total over n days > threshold in search domain 
# (set as oct-feb)
# specify rainfall dataframe, n, threshold
plantdate<-function(test,n,threshold){
	#restrict to planting season				
	season<-test[test$month %in% c(10:12,1:2),]
	if(season$month[1] %in% 10:12& !is.na(season$month[1])){
		season$pastrain<-NA
	#calculate past rain (over n days)
	for(i in (n+1):(nrow(season)-30)){season$pastrain[i]<-sum(season$rain[(i-n):i])}
	sop<-season[season$pastrain>threshold & season$pastrain!=season$rain & !is.na(season$pastrain),]
	firstplant<-sop$date[1]
	}else{
		firstplant<-NA}
return(as.Date(firstplant,origin='1970-01-01'))}


# season end date: first date where total rainfall over "past" days less than "thresholdpast" and total rainfall
# over "future" days less than "thresholdfuture"
enddate<-function(df,past,future,thresholdpast,thresholdfuture){
		season<-df[df$month %in% c(3:9),]
		if(nrow(season)>0){
		season$pastrain<-NA
		season$futurerain<-NA
	#calculate past rain (over n days)
	for(i in (past+1):nrow(season))season$pastrain[i]<-sum(season$rain[(i-past):i])
	#caluclate future rain over 20 days
	for(i in 1:(nrow(season)-future))season$futurerain[i]<-sum(season$rain[i:(i+future)])
	endop<-season[season$pastrain<thresholdpast & season$futurerain<thresholdfuture & !is.na(season$pastrain),]
	end<-endop$date[1]}else{end<-NA}
return(end)
}


# calculate start and end dates for each year
start<-ddply(met1,.(Hyear),function(df) startdate(df,2,30))
names(start)[2]<-'start.date'
end<-ddply(met1,.(Hyear),function(df) enddate(df,10,10,10,5))
names(end)[2]<-'end.date'


# set a removal period based on season start [calculated from above]. 
# start and end of removal period: in this case from seson start to 30 days after season start
period.dates<-data.frame(start=start$start.date,end=start$start.date+30)

# convert to julian days (day of year) and remove the raindays that define season start 
# (otherwise planting date won't change)
period.doy<-data.frame(Hyear=ifelse(strftime(period.dates$start,'%m') %in% 10:12,as.numeric(strftime(period.dates$start,'%Y'))+1,as.numeric(strftime(period.dates$start,'%Y'))),start=as.numeric(strftime(period.dates$start,'%j'))-2,end=as.numeric(strftime(period.dates$end,'%j')))

# remove years where start date doesn't exist [partial years in record]
period.doy<-period.doy[!is.na(period.doy$Hyear),]

# function to extract portions of a dataframe based on a period.doy type dataframe that contains start and 
# end days of year (to define a period) and Hyear column
extract.period<-function(met1,period.doy){
	hy<-met1$Hyear[1]
	start<-period.doy$start[period.doy$Hyear==hy&!is.na(period.doy$Hyear)]
	end<-period.doy$end[period.doy$Hyear==hy&!is.na(period.doy$Hyear)]
	met1.period<-met1[met1$doy %in% start:end,]
	met1.period}

# use extract.period for each year to get a full dataset containing only required periods.
start.period.met<-ddply(met1[met1$Hyear %in% period.doy$Hyear,],.(Hyear),function(df) extract.period(df,period.doy))

# create dataframe for non-treated period
non.start.period.met<-met1[!(met1$date %in% start.period.met$date),]

# add dummy columns to dataframe that will not be changed
non.start.period.met$count<-NA
non.start.period.met$rainOut<-non.start.period.met$rain
non.start.period.met$rainReduced<-0

# apply ChangeRainAmt to period file, then combine start.period.met and non.start.period.met, and 
# order correctly by date. Other removal and redistribution functions
# can be used as well and will work the same way
startout<-ddply(start.period.met,.(year),function(df)ChangeRainAmt(df,5,0,0.85,100))

# recombine period with reduced rainfall and period that is unchanged.
start.reduced<-rbind(non.start.period.met,startout)
# reorder by date
start.order<-start.reduced[with(start.reduced,order(date)),]

##
# similarly for removal before season end:
# find season end for each year
end<-ddply(met1,.(Hyear),function(df) enddate(df,10,10,10,5))
names(end)[2]<-'end.date'

# calculate and extract period on which to apply removal and redistribution functions
period.dates.end<-data.frame(start=end$end.date-30,end=end$end.date)
period.doy.end<-data.frame(Hyear=ifelse(strftime(period.dates.end$start,'%m') %in% 10:12,as.numeric(strftime(period.dates.end$start,'%Y'))+1,as.numeric(strftime(period.dates.end$start,'%Y'))),start=as.numeric(strftime(period.dates.end$start,'%j')),end=as.numeric(strftime(period.dates.end$end,'%j')))
period.doy.end<-period.doy.end[!is.na(period.doy.end$Hyear),]
end.period.met<-ddply(met1,.(Hyear),function(df) extract.period(df,period.doy.end))

# create dataframe for non-treated period
non.end.period.met<-met1[!(met1$date %in% end.period.met$date),]
non.end.period.met$count<-NA
non.end.period.met$rainOut<-non.end.period.met$rain
non.end.period.met$rainReduced<-0

# apply removal function within desired period
endout<-ddply(end.period.met,.(year),function(df)ChangeRainAmt(df,1,0,0.85,100))
# recombine treated and non-treated dataframes
end.reduced<-rbind(non.end.period.met,endout)
# reorder by date
end.order<-end.reduced[with(end.reduced,order(date)),]

############################
# Example metfile creation #
############################

# read in met data file
met1<-read.table('Kasungu.met',skip=9)
names(met1)<-c('year','doy','radn','maxt','mint','rain')
# add date and month columns for later calculations
met1$date<-as.Date(paste(met1$year,met1$doy,sep='-'),format='%Y-%j')
met1$month<-as.numeric(strftime(met1$date,'%m'))
met1$Hyear<-ifelse(met1$month %in% 10:12,met1$year+1,met1$year)

#remove rainfall 100% on 1 in 5 rainy days
dayOut<-ddply(met1,.(year),function(df)ChangeRainAmt(df,5,0,0.85,100))
#redistribute on 1 in 5 rainy days for total reduction of 10%
redistNROut<-ddply(dayOut,.(year),function(df) Redistribute.Nrain(df,0.85,5,0.1))

#set met as output from rainfall scenarios
met<-redistNROut
#modify temps (if needed)
met$maxt<-met$maxt+1
met$mint<-met$mint+1

# extract desired columns including adjusted rainfall
met.out<-met[,c(1:5,13)]
names(met.out)[2]<-'day'
names(met.out)[6]<-'rain'
met.calc<-met

# calculate mean temperature and amplitude in mean monthly temperature (needed for APSIM)
met.calc$tmean<-(met.calc$mint+met.calc$maxt)/2
monthly.avg.temp<-ddply(met.calc,.(month),function(df) mean(df$tmean))
tav<-mean(monthly.avg.temp$V1)
amp<-max(monthly.avg.temp$V1)-min(monthly.avg.temp$V1)

# create header information
head1<-paste(names(met.out),sep='\t')
head2<-paste(c("()","()","(MJ/m^2)","(oC)","(oC)","(mm)"),sep='\t')
header=as.data.frame(rbind(head1,head2))

# write to file with APSIM syntax
con <- file('kC1.met', 'w') 
writeLines(c("! Kasungu metfile", "[weather.met.weather]", "Latitude=-11.33",""), con = con,sep="\r\n")
cat(c("tav=",tav,"\r\n","amp=",amp,"\r\n","\r\n"),file=con) 
# output as fixed width for APSIM
write.fwf(header,rownames=F,colnames=F,file=con,width=10,eol="\r\n")
write.fwf(met.out,rownames=F,colnames=F,quote=F, file = con,width=10,eol="\r\n") 
close(con) 

